
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: mediamarks_favorites.c 1372 2006-10-16 10:39:45Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "codeset.h"
#include "environment.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "mediamarks.h"
#include "mediamarks_favorites.h"
#include "oxine.h"

#define OXINE_FAVORITES_XML "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
#define OXINE_FAVORITES_DTD "<!DOCTYPE oxine_favorites SYSTEM \"http://oxine.sf.net/DTD/favorites.dtd\">\n\n"

static int
favorites_swap_cb (void *d1, void *d2)
{
    fileitem_t *f1 = (fileitem_t *) d1;
    fileitem_t *f2 = (fileitem_t *) d2;

    return (f2->listened_count > f1->listened_count);
}


void
favorites_add_directory (filelist_t * favorites_filelist,
                         const char *title, const char *mrl)
{
    struct stat filestat;
    stat (mrl, &filestat);
    if (!S_ISDIR (filestat.st_mode)) {
        debug ("This is not a directory (ignoring): '%s'!", mrl);
        return;
    }

    char *rmrl = resolve_softlink (mrl);

    /* This is necessary, so the MRLs are always the same. */
    int len = strlen (rmrl);
    if (rmrl[len - 1] == '/')
        rmrl[len - 1] = 0;

    /* Try to find an entry with the same MRL. */
    fileitem_t *current = filelist_first (favorites_filelist);
    while (current && !(strcmp (current->mrl, rmrl) == 0)) {
        current = filelist_next (favorites_filelist, current);
    }

    /* If no entry was found, we add a new one. */
    if (!current) {
        current = filelist_add (favorites_filelist,
                                title, rmrl, FILE_TYPE_DIRECTORY);
    }

    ho_free (rmrl);
    assert (current);
    current->listened_count++;

    filelist_sort (favorites_filelist, favorites_swap_cb);

    /* Write the list back to the favorites file. */
    const char *fav_mrl = get_file_favorites ();
    FILE *f = fopen (fav_mrl, "w");
    if (f == NULL) {
        error (_("Could not open '%s': %s!"), fav_mrl, strerror (errno));
        return;
    }

    char *codeset = get_system_encoding ();
    recode_t *xr = recode_init (codeset, "utf8");

    fprintf (f, OXINE_FAVORITES_XML);
    fprintf (f, OXINE_FAVORITES_DTD);
    fprintf (f, "<!--\n");
    fprintf (f, "***************************************"
             "**************************************\n\n");
    fprintf (f, "This file is automaticaly created by oxine. "
             "Any changes made to this file\nwill be lost!\n\n");
    fprintf (f, "***************************************"
             "**************************************\n");
    fprintf (f, "-->\n\n");
    fprintf (f, "<oxine_favorites>\n");
    current = filelist_first (favorites_filelist);
    while (current) {

        if (current->type == FILE_TYPE_DIRECTORY) {
            char *utf8_mrl = recode (xr, current->mrl);
            char *utf8_title = recode (xr, current->title);
            fprintf (f, "  <entry>\n");
            fprintf (f, "    <mrl href=\"%s\" />\n", utf8_mrl);
            fprintf (f, "    <title>%s</title>\n", utf8_title);
            fprintf (f, "    <listened_count>%d</listened_count>\n",
                     current->listened_count);
            fprintf (f, "  </entry>\n");
            ho_free (utf8_mrl);
            ho_free (utf8_title);
        }

        current = filelist_next (favorites_filelist, current);
    }
    fprintf (f, "</oxine_favorites>\n");
    fclose (f);

    recode_done (xr);
    ho_free (codeset);
}


filelist_t *
favorites_new (filelist_t * toplevel_filelist)
{
    const char *mrl = get_file_favorites ();
    char *title = create_title (mrl);
    fileitem_t *fileitem = filelist_add (toplevel_filelist, title, mrl,
                                         FILE_TYPE_MEDIAMARKS);
    ho_free (title);

    filelist_expand (fileitem);

    filelist_sort (fileitem->sublist, favorites_swap_cb);
    filelist_sort (toplevel_filelist, NULL);

    return fileitem->sublist;
}
