
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: event.c 1571 2006-11-19 17:54:55Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdlib.h>

#include "environment.h"
#include "event.h"
#include "gui_utils.h"
#include "filelist_menu.h"
#include "heap.h"
#include "help_menu.h"
#include "i18n.h"
#include "logger.h"
#include "main_menu.h"
#include "oxine.h"
#include "playback_menu.h"
#include "scheduler.h"

static void
do_not_restart_playlist (oxine_t * oxine)
{
    if (oxine->current_playlist == oxine->rw_playlist) {
        playlist_set_current (oxine->rw_playlist, NULL);
    }
}


static void
oxine_dvd_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    if (ev->source.key == OXINE_KEY_NULL)
        return;
    if (!odk_current_is_playback_mode (oxine->odk))
        return;
    if (!odk_current_is_dvd (oxine->odk))
        return;

    /* we backup the key */
    oxine_key_id_t key_save = ev->source.key;
    /* and consume this event */
    ev->source.key = OXINE_KEY_NULL;

    switch (key_save) {
    case OXINE_KEY_ACTIVATE:
        hide_user_interface (oxine);
        break;
    case OXINE_KEY_PREV:
        /* This is necessary because we don't automatically forward events to
         * oxine when the GUI is visible. */
        if (!odk_get_forward_events_to_xine (oxine->odk)
            && odk_current_has_chapters (oxine->odk)) {
            odk_xine_event_send (oxine->odk, true, XINE_EVENT_INPUT_PREVIOUS);
        }
        break;
    case OXINE_KEY_NEXT:
        /* This is necessary because we don't automatically forward events to
         * oxine when the GUI is visible. */
        if (!odk_get_forward_events_to_xine (oxine->odk)
            && odk_current_has_chapters (oxine->odk)) {
            odk_xine_event_send (oxine->odk, true, XINE_EVENT_INPUT_NEXT);
        }
        break;
    default:
        /* As this was not an event we want to handle, we restore the original
         * key code. */
        ev->source.key = key_save;
        break;
    }
}


#ifdef HAVE_VDR
static void
oxine_vdr_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    if (ev->source.key == OXINE_KEY_NULL)
        return;
    if (!odk_current_is_playback_mode (oxine->odk))
        return;
    if (!odk_current_is_vdr (oxine->odk))
        return;

    /* we backup the key */
    oxine_key_id_t key_save = ev->source.key;
    /* and consume this event */
    ev->source.key = OXINE_KEY_NULL;

    switch (key_save) {
    case OXINE_KEY_ACTIVATE:
    case OXINE_KEY_MENU_CURRENT_TITLE:
    case OXINE_KEY_PLAY_VDR:
    case OXINE_KEY_SKIP:
    case OXINE_KEY_SEEK:
    case OXINE_KEY_SPEED:
    case OXINE_KEY_PLAYMODE:
    case OXINE_KEY_SPU_CHANNEL:
    case OXINE_KEY_SPU_OFFSET:
    case OXINE_KEY_AUDIO_OFFSET:
    case OXINE_KEY_ASPECT_RATIO:
    case OXINE_KEY_DEINTERLACE:
    case OXINE_KEY_EJECT:
    case OXINE_KEY_HELP:
        /* These keys are ignored when in VDR mode. */
        break;
    case OXINE_KEY_SELECT:
    case OXINE_KEY_PPLAY:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_PAUSE);
        break;
    case OXINE_KEY_PLAY:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_PLAY);
        break;
    case OXINE_KEY_PAUSE:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_PAUSE);
        break;
    case OXINE_KEY_STOP:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_STOP);
        break;
    case OXINE_KEY_FASTRWD:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_FASTREW);
        break;
    case OXINE_KEY_FASTFWD:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_FASTFWD);
        break;
    case OXINE_KEY_PREV:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_CHANNELMINUS);
        break;
    case OXINE_KEY_NEXT:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_CHANNELPLUS);
        break;
    case OXINE_KEY_BACK:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_BACK);
        break;
    case OXINE_KEY_VOLUME:
        if (ev->data.how > 0)
            odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_VOLPLUS);
        else
            odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_VOLMINUS);
        break;
    case OXINE_KEY_VOLMUTE:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_MUTE);
        break;
    case OXINE_KEY_MENU_OSD:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_INPUT_MENU1);
        break;
    case OXINE_KEY_AUDIO_CHANNEL:
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_AUDIO);
        break;
    case OXINE_KEY_MENU_MAIN:
    case OXINE_KEY_PLAY_DVB:
    case OXINE_KEY_PLAY_V4L:
    case OXINE_KEY_MENU_VIDEO:
    case OXINE_KEY_MENU_MUSIC:
    case OXINE_KEY_QUIT:
    case OXINE_KEY_SHUTDOWN:
        /* As we do not want to use the oxine OSD over VDR, we always stop TV
         * playback before showing our menus. */
        odk_xine_event_send (oxine->odk, false, XINE_EVENT_VDR_STOP);
        odk_stop (oxine->odk);
    default:
        /* As this was not an event we want to pass to VDR, we restore the
         * original key code. */
        ev->source.key = key_save;
        break;
    }
}
#endif


/* 
 * ***************************************************************************
 * Name:            oxine_stream_speed_key_handler
 * Access:          private
 *
 * Description:     Handles all stream speed related keypress events.
 * ***************************************************************************
 */
static void
oxine_stream_speed_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    if (ev->source.key == OXINE_KEY_NULL)
        return;
    if (!odk_current_is_playback_mode (oxine->odk))
        return;

    int cur_speed = odk_get_stream_param (oxine->odk, ODK_PARAM_SPEED);
    int new_speed = cur_speed;

    if ((oxine->current_menu == hide_user_interface)
        || (oxine->current_menu == show_playback_menu_cb)
        || (oxine->current_menu == show_playback_controls_cb)
        || (oxine->current_menu == show_playback_info_cb)) {
        switch (ev->source.key) {
        case OXINE_KEY_SELECT:
            if (cur_speed == ODK_SPEED_PAUSE)
                new_speed = ODK_SPEED_NORMAL;
            else
                new_speed = ODK_SPEED_PAUSE;
            break;
        case OXINE_KEY_SPEED:
            if (ev->data.how > 0) {
                if (cur_speed < ODK_SPEED_FAST_4)
                    new_speed <<= 1;
            } else {
                if (cur_speed > ODK_SPEED_SLOW_4)
                    new_speed >>= 1;
            }
            break;
        default:
            break;
        }
    }

    {
        switch (ev->source.key) {
        case OXINE_KEY_PPLAY:
            if (cur_speed == ODK_SPEED_PAUSE)
                new_speed = ODK_SPEED_NORMAL;
            else
                new_speed = ODK_SPEED_PAUSE;
            break;
        case OXINE_KEY_PLAY:
            new_speed = ODK_SPEED_NORMAL;
            break;
        case OXINE_KEY_PAUSE:
            new_speed = ODK_SPEED_PAUSE;
            break;
        default:
            break;
        }
    }

    if (cur_speed != new_speed) {
        odk_set_stream_param (oxine->odk, ODK_PARAM_SPEED, new_speed);
        show_stream_parameter (oxine, ODK_PARAM_SPEED);
    }
}


/* 
 * ***************************************************************************
 * Name:            oxine_playback_key_handler
 * Access:          private
 *
 * Description:     Handles all playback related keypress events.
 * ***************************************************************************
 */
static void
oxine_playback_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    if (!odk_current_is_playback_mode (oxine->odk))
        return;

    /* Evaluate all keys used by VDR. */
#ifdef HAVE_VDR
    oxine_vdr_key_handler (oxine, ev);
#endif

    /* Evaluate all keys used by DVDs. */
    oxine_dvd_key_handler (oxine, ev);

    /* Evaluate all the keys related only to stream speed. */
    oxine_stream_speed_key_handler (oxine, ev);

    if (ev->source.key == OXINE_KEY_NULL)
        return;

    /* First of all we handle all keys that are allowed anywhere. */
    switch (ev->source.key) {
    case OXINE_KEY_ACTIVATE:
        if (!oxine->stream_parameter_is_visible)
            show_stream_parameter (oxine, ODK_PARAM_SPEED);
        else
            hide_user_interface (oxine);
        break;
    case OXINE_KEY_MENU_CURRENT_TITLE:
        show_playback_menu_cb (oxine);
        break;
    case OXINE_KEY_MENU_OSD:
        if (!odk_current_has_video (oxine->odk))
            return;
        if (!oxine->user_interface_is_visible) {
            show_playback_menu_cb (oxine);
        } else {
            oxine->backto_menu = hide_user_interface;
            hide_user_interface (oxine);
        }
        break;
    case OXINE_KEY_VOLUME:
        odk_change_stream_param (oxine->odk, ODK_PARAM_AUDIO_VOLUME,
                                 ev->data.how, 0, 100);
        show_stream_parameter (oxine, ODK_PARAM_AUDIO_VOLUME);
        break;
    case OXINE_KEY_VOLMUTE:
        odk_change_stream_param (oxine->odk, ODK_PARAM_AUDIO_MUTE, 1, 0, 1);
        show_stream_parameter (oxine, ODK_PARAM_AUDIO_MUTE);
        break;
    case OXINE_KEY_STOP:
        odk_stop (oxine->odk);
        break;
    case OXINE_KEY_FASTFWD:
        odk_set_stream_param (oxine->odk, ODK_PARAM_SPEED, ODK_SPEED_FAST_2);
        show_stream_parameter (oxine, ODK_PARAM_SPEED);
        break;
    case OXINE_KEY_FASTRWD:
        warn ("xine-lib does not yet support fast rewind.");
        break;
    case OXINE_KEY_SKIP:
        odk_set_stream_param (oxine->odk, ODK_PARAM_SPEED, ODK_SPEED_NORMAL);
        odk_change_stream_param (oxine->odk, ODK_PARAM_POSITION,
                                 ev->data.how, 0, 0);
        show_stream_parameter (oxine, ODK_PARAM_POSITION);
        break;
    case OXINE_KEY_SEEK:
        odk_set_stream_param (oxine->odk, ODK_PARAM_SPEED, ODK_SPEED_NORMAL);
        odk_set_stream_param (oxine->odk, ODK_PARAM_POSITION, ev->data.how);
        show_stream_parameter (oxine, ODK_PARAM_POSITION);
        break;
    case OXINE_KEY_PREV:
        if (playlist_get_current_pos (oxine->current_playlist) > 0) {
            playlist_play_prev (oxine);
        }
        break;
    case OXINE_KEY_NEXT:
        if ((playlist_get_current_pos (oxine->current_playlist) + 1) !=
            playlist_length (oxine->current_playlist)) {
            playlist_play_next (oxine);
        }
        break;
    case OXINE_KEY_PLAYMODE:
        switch (playlist_get_playmode (oxine->current_playlist)) {
        case PLAYLIST_MODE_NORMAL:
            playlist_set_playmode (oxine->current_playlist,
                                   PLAYLIST_MODE_REPEAT);
            break;
        case PLAYLIST_MODE_REPEAT:
            playlist_set_playmode (oxine->current_playlist,
                                   PLAYLIST_MODE_RANDOM);
            break;
        case PLAYLIST_MODE_RANDOM:
            playlist_set_playmode (oxine->current_playlist,
                                   PLAYLIST_MODE_NORMAL);
            break;
        }
        current_menu_cb (oxine);
    default:
        break;
    }

    if (!odk_current_has_video (oxine->odk))
        return;

    /* Now we handle all keys that are only 
     * interesting if the stream has video. */
    switch (ev->source.key) {
    case OXINE_KEY_ZOOM:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_X,
                                 ev->data.how, 0, 400);
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_Y,
                                 ev->data.how, 0, 400);
        show_stream_parameter (oxine, ODK_PARAM_VO_ZOOM_X);
        break;
    case OXINE_KEY_ZOOM_X:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_X,
                                 ev->data.how, 0, 400);
        show_stream_parameter (oxine, ODK_PARAM_VO_ZOOM_X);
        break;
    case OXINE_KEY_ZOOM_Y:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_Y,
                                 ev->data.how, 0, 400);
        show_stream_parameter (oxine, ODK_PARAM_VO_ZOOM_X);
        break;
    case OXINE_KEY_ZOOM_RESET:
        odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_X, 100);
        odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ZOOM_Y, 100);
        show_stream_parameter (oxine, ODK_PARAM_VO_ZOOM_X);
        break;
    case OXINE_KEY_SATURATION:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_SATURATION,
                                 ev->data.how, 0, 65535);
        show_stream_parameter (oxine, ODK_PARAM_VO_SATURATION);
        break;
    case OXINE_KEY_BRIGHTNESS:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_BRIGHTNESS,
                                 ev->data.how, 0, 65535);
        show_stream_parameter (oxine, ODK_PARAM_VO_BRIGHTNESS);
        break;
    case OXINE_KEY_CONTRAST:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_CONTRAST,
                                 ev->data.how, 0, 65535);
        show_stream_parameter (oxine, ODK_PARAM_VO_CONTRAST);
        break;
    case OXINE_KEY_HUE:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_HUE,
                                 ev->data.how, 0, 65535);
        show_stream_parameter (oxine, ODK_PARAM_VO_HUE);
        break;
    case OXINE_KEY_SPU_CHANNEL:
        {
            int max_channels = odk_get_stream_info (oxine->odk,
                                                    ODK_STREAM_INFO_MAX_SPU_CHANNEL);
            if (max_channels > 1) {
                odk_change_stream_param (oxine->odk,
                                         ODK_PARAM_SPU_CHANNEL,
                                         ev->data.how, -2, max_channels - 1);
                show_stream_parameter (oxine, ODK_PARAM_SPU_CHANNEL);
            }
        }
        break;
    case OXINE_KEY_SPU_OFFSET:
        odk_change_stream_param (oxine->odk, ODK_PARAM_SPU_OFFSET,
                                 ev->data.how, -90000, 90000);
        show_stream_parameter (oxine, ODK_PARAM_SPU_OFFSET);
        break;
    case OXINE_KEY_AUDIO_CHANNEL:
        {
            int max_channels = odk_get_stream_info (oxine->odk,
                                                    ODK_STREAM_INFO_MAX_AUDIO_CHANNEL);
            if (max_channels > 1) {
                odk_change_stream_param (oxine->odk,
                                         ODK_PARAM_AUDIO_CHANNEL,
                                         ev->data.how, -2, max_channels - 1);
                show_stream_parameter (oxine, ODK_PARAM_AUDIO_CHANNEL);
            }
        }
        break;
    case OXINE_KEY_AUDIO_OFFSET:
        odk_change_stream_param (oxine->odk,
                                 ODK_PARAM_AUDIO_OFFSET,
                                 ev->data.how, -90000, 90000);
        show_stream_parameter (oxine, ODK_PARAM_AUDIO_OFFSET);
        break;
    case OXINE_KEY_ASPECT_RATIO:
        odk_change_stream_param (oxine->odk,
                                 ODK_PARAM_VO_ASPECT_RATIO, 1,
                                 ODK_VO_ASPECT_AUTO, ODK_VO_ASPECT_DVB);
        show_stream_parameter (oxine, ODK_PARAM_VO_ASPECT_RATIO);
        break;
    case OXINE_KEY_DEINTERLACE:
        odk_change_stream_param (oxine->odk, ODK_PARAM_VO_DEINTERLACE,
                                 1, 0, 1);
        show_stream_parameter (oxine, ODK_PARAM_VO_DEINTERLACE);
        break;
    default:
        break;
    }
}


/* 
 * ***************************************************************************
 * Name:            oxine_key_handler
 * Access:          private
 *
 * Description:     Handles keypress events.
 * ***************************************************************************
 */
static void
oxine_key_handler (oxine_t * oxine, oxine_event_t * ev)
{
    /* Evaluate all playback related keypress events. */
    oxine_playback_key_handler (oxine, ev);

    if (ev->source.key == OXINE_KEY_NULL)
        return;

    switch (ev->source.key) {
    case OXINE_KEY_BACK:
        backto_menu_cb (oxine);
        break;
    case OXINE_KEY_HELP:
        show_help_menu_cb (oxine);
        break;
    case OXINE_KEY_EJECT:
        eject_cb (oxine);
        break;
    case OXINE_KEY_MENU_MAIN:
        show_main_menu_cb (oxine);
        break;
#ifdef HAVE_VDR
    case OXINE_KEY_PLAY_VDR:
        do_not_restart_playlist (oxine);
        play_vdr_cb (oxine);
        break;
#endif
    case OXINE_KEY_PLAY_DVB:
        do_not_restart_playlist (oxine);
        play_dvb_cb (oxine);
        break;
    case OXINE_KEY_PLAY_V4L:
        do_not_restart_playlist (oxine);
        play_v4l_cb (oxine);
        break;
    case OXINE_KEY_MENU_VIDEO:
        enter_video_menu_cb (oxine);
        break;
    case OXINE_KEY_MENU_MUSIC:
        enter_music_menu_cb (oxine);
        break;
    case OXINE_KEY_FULLSCREEN:
        if (odk_is_fullscreen (oxine->odk))
            odk_set_fullscreen (oxine->odk, 0);
        else
            odk_set_fullscreen (oxine->odk, 1);
        break;
    case OXINE_KEY_QUIT:
        odk_exit (oxine->odk);
        break;
    case OXINE_KEY_SHUTDOWN:
        shutdown_cb (oxine);
        break;
    default:
        break;
    }
}


/* 
 * ***************************************************************************
 * Name:            oxine_motion_handler
 * Access:          public
 *
 * Description:     Handles mouse movements
 * ***************************************************************************
 */
static void
oxine_motion_handler (oxine_t * oxine, oxine_event_t * event)
{
    if (odk_current_is_playback_mode (oxine->odk)
#ifdef HAVE_VDR
        && !odk_current_is_vdr (oxine->odk)
#endif
        && ((oxine->current_menu == hide_user_interface)
            || (oxine->current_menu == show_playback_info_cb)
            || (oxine->current_menu == show_playback_controls_cb))) {

        if (!oxine->playback_controls_are_visible
            && (event->data.pos.y > 550)) {
            show_playback_controls_cb (oxine);
        }

        if (oxine->playback_controls_are_visible && (event->data.pos.y < 500)) {
            hide_user_interface (oxine);
        }

        if (oxine->playback_controls_are_visible && (event->data.pos.y > 500)) {
            cancel_job (oxine->otk_clear_job);
            oxine->otk_clear_job = schedule_job (6000, (void (*)(void *))
                                                 hide_user_interface, oxine);
        }
    }
}


/* 
 * ***************************************************************************
 * Name:            oxine_event_handler
 * Access:          public
 *
 * Description:     This is the main event handler.
 * ***************************************************************************
 */
void
oxine_event_handler (void *oxine_p, oxine_event_t * event)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

#ifdef DEBUG_EVENTS
    switch (event->type) {
    case OXINE_EVENT_KEY:
        debug ("got OXINE_EVENT_KEY %d", event->source.key);
        break;
    case OXINE_EVENT_BUTTON:
        debug ("got OXINE_EVENT_BUTTON %d +%d+%d", event->source.button,
               event->data.pos.x, event->data.pos.y);
        break;
    case OXINE_EVENT_MOTION:
        debug ("got OXINE_EVENT_MOTION +%d+%d", event->data.pos.x,
               event->data.pos.y);
        break;
    case OXINE_EVENT_PLAYBACK_STARTED:
        debug ("got OXINE_EVENT_PLAYBACK_STARTED");
        break;
    case OXINE_EVENT_PLAYBACK_STOPPED:
        debug ("got OXINE_EVENT_PLAYBACK_STOPPED");
        break;
    case OXINE_EVENT_PLAYBACK_FINISHED:
        debug ("got OXINE_EVENT_PLAYBACK_FINISHED");
        break;
    default:
        break;
    }
#endif

    /* When we're in playback mode and are showing a menu that is not the
     * playing menu we start a timeout of 60 secs. after which we hide the
     * user interface if we're playing a stream with video or show the playing
     * menu if it's an audio only stream. */
    cancel_job (oxine->playback_menu_job);
    if (odk_current_is_playback_mode (oxine->odk)
        && oxine->user_interface_is_visible
        && (oxine->current_menu != show_playback_menu_cb)) {
        if (odk_current_has_video (oxine->odk)) {
            oxine->playback_menu_job =
                schedule_job (60 * 1000, (void (*)(void *))
                              hide_user_interface, oxine);
        } else {
            oxine->playback_menu_job =
                schedule_job (60 * 1000, (void (*)(void *))
                              show_playback_menu_cb, oxine);
        }
    }

    switch (event->type) {
    case OXINE_EVENT_KEY:
        oxine_key_handler (oxine, event);
        break;
    case OXINE_EVENT_MOTION:
        oxine_motion_handler (oxine, event);
        break;
    case OXINE_EVENT_PLAYBACK_STARTED:
        if (!odk_current_is_playback_mode (oxine->odk))
            break;
#ifdef HAVE_VDR
        if (odk_current_is_vdr (oxine->odk)) {
            hide_user_interface (oxine);
            break;
        }
#endif

        if ((oxine->current_menu != hide_user_interface)
            && (oxine->current_menu != show_playback_menu_cb)) {
            current_menu_cb (oxine);
            break;
        }

        if (!odk_current_has_video (oxine->odk)) {
            show_playback_menu_cb (oxine);
        } else if (odk_current_is_v4l (oxine->odk)
                   || odk_current_is_dvb (oxine->odk)
                   || odk_current_is_dvd (oxine->odk)
                   || odk_current_is_vcd (oxine->odk)) {
            hide_user_interface (oxine);
        } else {
            show_playback_info_cb (oxine);
        }
        break;
    case OXINE_EVENT_PLAYBACK_STOPPED:
        do_not_restart_playlist (oxine);
        playback_ended_menu_cb (oxine);
        break;
    case OXINE_EVENT_PLAYBACK_FINISHED:
        playlist_play_next (oxine);
        break;
    case OXINE_EVENT_PLAYBACK_ERROR:
        show_message_dialog ((otk_cb_t) playlist_play_next, oxine,
                             (otk_cb_t) playback_ended_menu_cb, oxine,
                             DIALOG_OK_CANCEL, NULL,
                             _("Could not play title!"));
        break;
    case OXINE_EVENT_WAITING_FOR_ALTERNATIVE:
        show_message_dialog (NULL, NULL, NULL, NULL, DIALOG_PLAIN, NULL,
                             _("Waiting for alternative stream..."));
        break;
    default:
        break;
    }
}
