#include "grail-fixture.h"

void GrailTest::SendFrameEvent(
    uint64_t time,
    UFWindowId window_id,
    const struct UFTouch_ &touch_struct)
{
  std::vector<struct UFTouch_> touches;
  touches.push_back(touch_struct);
  SendFrameEvent(time, window_id, touches);
}

void GrailTest::SendFrameEvent(
    uint64_t time,
    UFWindowId window_id,
    std::vector<struct UFTouch_> &touches)
{
  UFFrame frame = new struct UFFrame_;
  frame->device_ptr = device_ptr;
  frame->window_id = window_id;
  frame->touches = touches;

  UFEvent frame_event = new struct UFEvent_;
  frame_event->type = UFEventTypeFrame;
  frame_event->time = time;
  frame_event->frame_ptr.reset(std::move(frame));

  grail_process_frame_event(grail_handle, frame_event);
  frame_event_unref(frame_event);
  frame_event = nullptr;
}

void GrailTest::SendDeviceAddedEvent(uint64_t time)
{
  UFEvent frame_event = new struct UFEvent_;
  frame_event->type = UFEventTypeDeviceAdded;
  frame_event->time = time;
  frame_event->device_ptr = device_ptr;

  grail_process_frame_event(grail_handle, frame_event);
  frame_event_unref(frame_event);
  frame_event = nullptr;
}

void GrailTest::FillFakeDeviceStruct()
{
  struct UFAxis_ axis_x_struct;
  axis_x_struct.type = UFAxisTypeX;
  axis_x_struct.resolution = 3764.70;

  struct UFAxis_ axis_y_struct;
  axis_y_struct.type = UFAxisTypeY;
  axis_y_struct.resolution = 3764.70;

  device_ptr.reset(new struct UFDevice_);
  device_ptr->direct = 1;
  device_ptr->axes.push_back(axis_x_struct);
  device_ptr->axes.push_back(axis_y_struct);
  /* pixels/m */
  device_ptr->window_resolution_x = 3764.70;
  device_ptr->window_resolution_y = 3764.70;
}

UGSubscription GrailTest::CreateSubscription(
    unsigned int num_touches, UGGestureTypeMask gesture_mask,
    UFDevice device, UFWindowId window_id)
{
  UGStatus status;
  UGSubscription subscription = nullptr;

  status = grail_subscription_new(&subscription);
  if (status != UGStatusSuccess)
  {
    ADD_FAILURE() << "failed to create subscription";
    return nullptr;
  }

  status = grail_subscription_set_property(subscription,
                                           UGSubscriptionPropertyDevice,
                                           &device);
  if (status != UGStatusSuccess)
  {
    ADD_FAILURE() << "failed to set device subscription";
    grail_subscription_delete(subscription);
    return nullptr;
  }

  status = grail_subscription_set_property(subscription,
                                           UGSubscriptionPropertyWindow,
                                           &window_id);
  if (status != UGStatusSuccess)
  {
    ADD_FAILURE() << "failed to set subscription window";
    grail_subscription_delete(subscription);
    return nullptr;
  }

  status = grail_subscription_set_property(subscription,
                                           UGSubscriptionPropertyTouchesStart,
                                           &num_touches);
  if (status != UGStatusSuccess)
  {
    ADD_FAILURE() << "failed to set subscription start touches";
    grail_subscription_delete(subscription);
    return nullptr;
  }

  status = grail_subscription_set_property(subscription,
                                           UGSubscriptionPropertyTouchesMaximum,
                                           &num_touches);
  if (status != UGStatusSuccess)
  {
    ADD_FAILURE() << "failed to set subscription start touches";
    grail_subscription_delete(subscription);
    return nullptr;
  }

  status = grail_subscription_set_property(subscription,
                                           UGSubscriptionPropertyTouchesMinimum,
                                           &num_touches);
  if (status != UGStatusSuccess)
  {
    ADD_FAILURE() << "failed to set subscription min touches";
    grail_subscription_delete(subscription);
    return nullptr;
  }

  status = grail_subscription_set_property(subscription,
      UGSubscriptionPropertyMask, &gesture_mask);
  if (status != UGStatusSuccess)
  {
    ADD_FAILURE() << "failed to set subscription mask";
    grail_subscription_delete(subscription);
    return nullptr;
  }

  status = grail_subscription_activate(grail_handle, subscription);
  if (status != UGStatusSuccess)
  {
    ADD_FAILURE() << "failed to activate subscription";
    grail_subscription_delete(subscription);
    return nullptr;
  }

  return subscription;
}

void GrailTest::BeginTouch(int touch_id)
{
  BeginTouchWindowCoords(touch_id, touch_id * 10.0f, 0.0f);
}

void GrailTest::BeginTouchWindowCoords(int touch_id, float window_x, float window_y)
{
  struct UFTouch_ touch_struct;
  touch_struct.id = touch_id;
  touch_struct.state = UFTouchStateBegin;
  touch_struct.time = time;
  touch_struct.start_time = touch_struct.time;
  touch_struct.window_x = window_x;
  touch_struct.window_y = window_y;
  touch_struct.owned = 0;
  touch_struct.pending_end = 0;

  touches.push_back(touch_struct);

  SendFrameEvent(time, fake_window_id, touches);

  // leave it in a neutral state for next uses of touches vector
  touches[touches.size()-1].state = UFTouchStateUpdate;
}

void GrailTest::GiveTouchOwnership(int touch_id)
{
  auto iterator = FindTouch(touch_id);
  iterator->state = UFTouchStateUpdate;
  iterator->owned = 1; // ownership will trigger delivery of grail events.
  iterator->time = time;

  SendFrameEvent(time, fake_window_id, touches);
}

void GrailTest::SetTouchWindowCoords(int touch_id, float window_x, float window_y)
{
  auto iterator = FindTouch(touch_id);
  iterator->window_x = window_x;
  iterator->window_y = window_y;
}

void GrailTest::UpdateTouches()
{
  for (struct UFTouch_ &touch : touches)
  {
    touch.time = time;
  }
  SendFrameEvent(time, fake_window_id, touches);
}

void GrailTest::EndTouch(int touch_id)
{
  auto iterator = FindTouch(touch_id);
  iterator->state = UFTouchStateEnd;
  iterator->time = time;

  SendFrameEvent(time, fake_window_id, touches);

  touches.erase(iterator);
}

std::vector<struct UFTouch_>::iterator GrailTest::FindTouch(unsigned int touch_id)
{
  std::vector<struct UFTouch_>::iterator it = touches.begin();

  while (it != touches.end())
  {
    if (it->id == touch_id)
      return it;
    ++it;
  }

  return it;
}

void GrailTest::ProcessGrailEvents()
{
  UGEvent grail_event;
  UGStatus get_event_status;

  do
  {
    get_event_status = grail_get_event(grail_handle, &grail_event);
    if (get_event_status != UGStatusSuccess)
      continue;

    switch (grail_event_get_type(grail_event))
    {
      case UGEventTypeSlice:
        {
          UGStatus status;
          UGSlice slice;
          status = grail_event_get_property(grail_event, UGEventPropertySlice, &slice);
          ASSERT_EQ(UGStatusSuccess, status);
          ProcessSlice(slice);
        }
        break;
      default:
        FAIL(); // we are only expecting slice events
    }

    grail_event_unref(grail_event);
  }
  while (get_event_status == UGStatusSuccess);

  ASSERT_EQ(UGStatusErrorNoEvent, get_event_status);
}

void GrailTest::ProcessSlice(UGSlice slice)
{
  switch (grail_slice_get_state(slice))
  {
    case UGGestureStateBegin:
      AddNewGesture(slice);
      break;
    case UGGestureStateUpdate:
      UpdateGesture(slice);
      break;
    case UGGestureStateEnd:
      EndGesture(slice);
      break;
    default:
      FAIL();
  }
}

GrailGesture *GrailTest::GestureWithId(unsigned int id)
{
  GrailGesture *gesture = nullptr;

  for (GrailGesture &other_gesture : grail_gestures)
  {
    if (id == other_gesture.id)
    {
      gesture = &other_gesture;
      break;
    }
  }

  return gesture;
}

void GrailTest::AddNewGesture(UGSlice slice)
{
  unsigned int gesture_id;
  UGStatus status =  grail_slice_get_property(slice, UGSlicePropertyId, &gesture_id);
  ASSERT_EQ(UGStatusSuccess, status);

  // there should be no gesture with this id
  ASSERT_EQ(nullptr, GestureWithId(gesture_id));

  GrailGesture gesture;
  gesture.id = gesture_id;

  unsigned int num_touches;
  status =  grail_slice_get_property(slice, UGSlicePropertyNumTouches, &num_touches);
  ASSERT_EQ(UGStatusSuccess, status);
  for (unsigned int i = 0; i < num_touches; ++i)
  {
    UFTouchId touch_id;
    status = grail_slice_get_touch_id(slice, i, &touch_id);
    ASSERT_EQ(UGStatusSuccess, status);
    gesture.touches.insert(touch_id);
  }

  gesture.construction_finished = grail_slice_get_construction_finished(slice);
  gesture.state = UGGestureStateBegin;

  grail_gestures.push_back(gesture);
}

void GrailTest::UpdateGesture(UGSlice slice)
{
  GrailGesture *gesture = nullptr;
  unsigned int gesture_id;

  UGStatus status = grail_slice_get_property(slice, UGSlicePropertyId, &gesture_id);
  ASSERT_EQ(UGStatusSuccess, status);

  gesture = GestureWithId(gesture_id);

  // the gesture must already exist
  ASSERT_NE(nullptr, gesture);

  if (grail_slice_get_construction_finished(slice))
  {
    gesture->construction_finished = true;
  }
  else
  {
    // can't go back from true to false
    ASSERT_FALSE(gesture->construction_finished);
  }

  ASSERT_TRUE(gesture->state == UGGestureStateBegin
           || gesture->state == UGGestureStateUpdate);
  gesture->state = UGGestureStateUpdate;
}

void GrailTest::EndGesture(UGSlice slice)
{
  GrailGesture *gesture = nullptr;
  unsigned int gesture_id;

  UGStatus status = grail_slice_get_property(slice, UGSlicePropertyId, &gesture_id);
  ASSERT_EQ(UGStatusSuccess, status);

  gesture = GestureWithId(gesture_id);

  // the gesture must already exist
  ASSERT_NE(nullptr, gesture);

  gesture->construction_finished = grail_slice_get_construction_finished(slice);
  ASSERT_TRUE(gesture->construction_finished);

  ASSERT_TRUE(gesture->state == UGGestureStateBegin
           || gesture->state == UGGestureStateUpdate);
  gesture->state = UGGestureStateEnd;
}

void GrailTest::PrintPendingGestures()
{
  UGEvent grail_event;
  UGStatus get_event_status;

  do
  {
    get_event_status = grail_get_event(grail_handle, &grail_event);
    if (get_event_status != UGStatusSuccess)
      continue;

    switch (grail_event_get_type(grail_event))
    {
      case UGEventTypeSlice:
        {
          UGStatus status;
          UGSlice slice;
          status = grail_event_get_property(grail_event, UGEventPropertySlice, &slice);
          ASSERT_EQ(UGStatusSuccess, status);
          PrintSlice(slice);
        }
        break;
      default:
        std::cout << "other event\n";
    }

    grail_event_unref(grail_event);
  }
  while (get_event_status == UGStatusSuccess);

  ASSERT_EQ(UGStatusErrorNoEvent, get_event_status);
}

void GrailTest::PrintSlice(UGSlice slice)
{
  UGStatus status;
  unsigned int num_touches;
  unsigned int gesture_id;

  std::cout << "slice";

  status =  grail_slice_get_property(slice, UGSlicePropertyId, &gesture_id);
  ASSERT_EQ(UGStatusSuccess, status);
  std::cout << " id=" << gesture_id;

  switch (grail_slice_get_state(slice))
  {
    case UGGestureStateBegin:
      std::cout << " begin";
      break;
    case UGGestureStateUpdate:
      std::cout << " update";
      break;
    case UGGestureStateEnd:
      std::cout << " end";
      break;
    default:
      FAIL();
  }

  status =  grail_slice_get_property(slice, UGSlicePropertyNumTouches, &num_touches);
  ASSERT_EQ(UGStatusSuccess, status);
  std::cout << " touches=";
  for (unsigned int i = 0; i < num_touches; ++i)
  {
    if (i != 0)
      std::cout << ",";

    UFTouchId touch_id;
    status = grail_slice_get_touch_id(slice, i, &touch_id);
    ASSERT_EQ(UGStatusSuccess, status);
    std::cout << touch_id;
  }


  if (grail_slice_get_construction_finished(slice))
    std::cout << " finished";
  else
    std::cout << " unfinished";

  std::cout << "\n";
}
