/* android-addressbook-plugin.c - Android addressbook plugin
 *
 * Copyright (C) 2012 Canonical, Ltd. (www.canonical.com)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors: Renato Araujo Oliveira Filho <renato@canonical.com>
 */

#include <glib/gi18n.h>

#include <libedataserver/eds-version.h>
#include <libedataserver/e-source.h>
#include <libedataserver/e-source-list.h>

#include <addressbook/gui/widgets/eab-config.h>

#include <libebook/e-book.h>

#include "android-addressbook-plugin.h"

#define ANDROID_BASE_URI "android://"

void
ensure_source_group (ESourceList *source_list)
{
    if (source_list == NULL) {
       return;
    }

    e_source_list_ensure_group(source_list, "Ubuntu Android", ANDROID_BASE_URI, FALSE);

    ESourceGroup *group = e_source_list_peek_group_by_base_uri (source_list, ANDROID_BASE_URI);
    GSList *sources = e_source_group_peek_sources(group);
    if (!sources || (g_slist_length(sources) == 0)) {
        ESource *source = e_source_new ("Android", "//");
        e_source_set_property(source, "completion", "true");

        e_source_group_add_source(group, source, -1);
        e_source_group_set_readonly(group, FALSE);

        e_source_list_sync (source_list, NULL);
        g_object_unref (source);
    }
}

void
remove_source_group (ESourceList *source_list)
{
    if (source_list) {
        ESourceGroup *group;

        group = e_source_list_peek_group_by_base_uri (source_list, ANDROID_BASE_URI);
        if (group) {
            GSList *sources;

            sources = e_source_group_peek_sources (group);
            if (sources == NULL) {
                e_source_list_remove_group (source_list, group);
                e_source_list_sync (source_list, NULL);
            }
        }

        g_object_unref (G_OBJECT (source_list));
    }
}

GtkWidget *
plugin_android_addressbook (EPlugin *epl, EConfigHookItemFactoryData *data)
{
    ESource *source;
    ESourceGroup *group;
    const gchar *base_uri;
    EABConfigTargetSource *t = (EABConfigTargetSource *) data->target;

    source = t->source;
    group  = e_source_peek_group (source);
    base_uri = e_source_group_peek_base_uri (group);

    g_object_set_data (G_OBJECT (epl), "cwidget", NULL);
    if (strcmp (base_uri, ANDROID_BASE_URI) != 0)
        return NULL;

    return NULL;
}


gint
e_plugin_lib_enable (EPlugin *ep, gint enable)
{
    ESourceList *book_sources;

    e_book_get_addressbooks (&book_sources, NULL);

    if (enable)
        ensure_source_group (book_sources);
    else
        remove_source_group (book_sources);

    return 0;
}

