/*
 * Enhanced Easy Network
 * Copyright (c) <2009>, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "XMLOperator.h"
#include "ENMUtil.h"
#include <wx/dir.h>
#include <wx/filename.h>
#if defined(__UNIX__)

const wxString XMLOperator::VERSIONFILEPATH =  wxT("/etc/Network Neighborhood/VersionInfo.xml");
const wxString XMLOperator::NNFILEPATH =  wxT("/etc/Network Neighborhood/");
const wxString XMLOperator::MESHCONFIGURATIONFILEPATH = wxT("MeshConfiguation.xml");

const wxString XMLOperator::SYSTEM_MESH_DIR = wxT("/etc/Network Neighborhood/");
const wxString XMLOperator::USER_MESH_DIR = ENMUtil::GetCurrentUserAppDataFolder() + _T("/Network Neighborhood/");
#endif


/*=================================================================
 Function Name  : XMLOperator::GetNNfilePath
 Description    : Return NN file path
 Date           : 2008-9-16 16:28:26
 Return Code    : wxString
 Author         : Guo xixiu
 =================================================================*/
wxString  XMLOperator::GetNNfilePath()
{
    return NNFILEPATH;
}


/*=================================================================
 Function Name  : XMLOperator::XMLOperator
 Description    : Constructor
 Date           : 2008-9-16 16:28:28
 Return Code    : None
 Author         : Guo xixiu
 =================================================================*/
 XMLOperator::XMLOperator()
{
    //ctor

}


/*=================================================================
 Function Name  : XMLOperator::~XMLOperator
 Description    : Destructor
 Date           : 2008-9-16 16:28:31
 Return Code    : None
 Author         : Guo xixiu
 =================================================================*/
XMLOperator::~XMLOperator()
{
    //dtor
}

/*=================================================================
 Function Name  : XMLOperator::CreateVersionInfoXmlFile
 Description    : Create profile/policy version information file
 Date           : 2008-9-16 16:29:21
 Parameter      : const wxArrayString& VersionInfo
 Return Code    : bool
 Author         : Guo xixiu
 =================================================================*/
bool  XMLOperator::CreateVersionInfoXmlFile(const wxArrayString& VersionInfo)
{
    //create root element
    wxXmlNode* pRoot = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, _T("VersionInfo"));
    pRoot->AddProperty(_T("version"), VersionInfo.Item(0));

    /*create File elements*/
    for (size_t i = 1; i < VersionInfo.GetCount(); i++)
    {
        wxXmlNode* pFileNode = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, _T("File"));
        if (NULL == pFileNode)
        {
            delete pRoot;
            return false;
        }
        pFileNode->AddProperty(_T("name"), VersionInfo.Item(i));
        pRoot->AddChild(pFileNode);
    }
    /*save the file*/
    if ( !wxDir::Exists(NNFILEPATH) && !::wxMkdir(NNFILEPATH) )
        return false;
    wxString filePath = VERSIONFILEPATH;
    wxXmlDocument doc;
	doc.SetVersion(_T("1.0"));
	doc.SetFileEncoding(_T("UTF-8"));
	doc.SetRoot(pRoot);
    return doc.Save(filePath);
}


/*=================================================================
 Function Name  :  XMLOperator::ReadVersionInfoXmlFile
 Description    : Read profile/policy version information from file
 Date           : 2008-9-16 16:29:25
 Parameter      : wxArrayString &VersionInfo
 Return Code    : bool
 Author         : Guo xixiu
 =================================================================*/
bool   XMLOperator::ReadVersionInfoXmlFile(wxArrayString &VersionInfo)
{
    wxString filePath = VERSIONFILEPATH;
    if ( !wxFileName::FileExists(filePath) )
    {
        return false;
    }
    //load xml file
    wxXmlDocument doc;
    if (!doc.Load(filePath))
    {
        return false;
    }

    //get root element
    wxXmlNode* pRoot = doc.GetRoot();
    if (NULL == pRoot)
    {
        return false;
    }

    if ( pRoot->GetName() != _T("VersionInfo") )
        return false;
    wxString version;
    if (!pRoot->GetPropVal(_T("version"), &version))
        return false;
    VersionInfo.Add(version);

    wxXmlNode* pChild = pRoot->GetChildren();
    bool bReturn = true;
    while ( NULL != pChild )
    {
        wxString filename;
        if ( pChild->GetName() == _T("File") )
        {
            bReturn &= pChild->GetPropVal(_T("name"), &filename);
            VersionInfo.Add(filename);
        }
        pChild = pChild->GetNext();
    }
    return bReturn;
}

bool   XMLOperator::ModifyVersionInfoXmlFile(const wxArrayString& VersionInfo)
{
    wxString filePath = VERSIONFILEPATH;
    if ( !wxFileName::FileExists(filePath) )
    {
        return false;
    }
    //load xml file
    wxXmlDocument doc;
    if (!doc.Load(filePath))
    {
        return false;
    }

    //get root element
    wxXmlNode* pRoot = doc.GetRoot();
    if (NULL == pRoot)
    {
        return false;
    }

    if ( pRoot->GetName() != _T("VersionInfo") )
        return false;

    wxXmlNode* pChild = pRoot->GetChildren();
    while ( NULL != pChild )
    {
        wxString filename;
        wxXmlNode* pBackUp = pChild->GetNext();
        if ( pChild->GetName() == _T("File") )
        {
            pChild->GetPropVal(_T("name"), &filename);
            for ( int i = 0; i < VersionInfo.Count(); i++ )
            {
                if ( filename == VersionInfo.Item(i) )
                    pRoot->RemoveChild(pChild);
            }
        }
        pChild = pBackUp;
    }
    /*save the file*/
    return doc.Save(filePath);

}


/*=================================================================
 Function Name  : XMLOperator::CreateMeshLocalHostNameXmlFile
 Description    : Create mesh local host name file
 Date           : 2008-9-16 16:28:53
 Parameter      : const wxString &strMeshLocalHostName
 Return Code    : bool
 Author         : liyang
 =================================================================*/
bool  XMLOperator::CreateMeshConfigurationXmlFile(MeshConfigurationEntity &meshConfiguration)
{

    //create root element
    wxXmlNode* pRoot = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, _T("Mesh"));

    //create MeshLocalHostName elemnet
    wxXmlNode* pMeshLocalHostName = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, _T("MeshLocalHostName"));
    if ( NULL == pMeshLocalHostName )
    {
        delete pRoot;
        return false;
    }
    pMeshLocalHostName->AddProperty(_T("value"), meshConfiguration.GetMeshLocalHostName());
    pRoot->AddChild(pMeshLocalHostName);

    //create MeshChannel element
    wxXmlNode* pMeshChannel = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, _T("MeshChannel"));
    if ( NULL == pMeshChannel )
    {
        delete pRoot;
        return false;
    }
    pMeshChannel->AddProperty(_T("value"), wxString::Format(_T("%d"), meshConfiguration.GetMeshChannel()));
    pRoot->AddChild(pMeshChannel);

    //create MeshAdvertiseTime element
    wxXmlNode* pMeshAdvertiseTime = new wxXmlNode(NULL, wxXML_ELEMENT_NODE, _T("MeshAdvertiseTime"));
    if ( NULL == pMeshAdvertiseTime )
    {
        delete pRoot;
        return false;
    }
    pMeshAdvertiseTime->AddProperty(_T("value"), wxString::Format(_T("%d"), meshConfiguration.GetMeshAdvertiseTime()));
    pRoot->AddChild(pMeshAdvertiseTime);

    //save the file
    bool isAdminUser = ENMUtil::IsAdministrator();
    wxString filePath;
    if (isAdminUser)
    {
        filePath = SYSTEM_MESH_DIR;
    }
    else
    {
        filePath = USER_MESH_DIR;
    }
    if ( !wxDir::Exists(filePath) && !::wxMkdir(filePath) )
        return false;

    wxXmlDocument doc;
    doc.SetVersion(_T("1.0"));
	doc.SetFileEncoding(_T("UTF-8"));
	doc.SetRoot(pRoot);

    return doc.Save(filePath + MESHCONFIGURATIONFILEPATH);

}


/*=================================================================
 Function Name  : XMLOperator::ReadMeshLocalHostNameXmlFile
 Description    : Read mesh local host name from file
 Date           : 2008-9-16 16:28:47
 Parameter      : wxString &strMeshLocalHostName
 Return Code    : bool
 Author         : liyang
 =================================================================*/
bool  XMLOperator::ReadMeshConfigurationXmlFile(MeshConfigurationEntity &meshConfiguration)
{
    //load xml file
    bool isAdminUser = ENMUtil::IsAdministrator();
    wxString filePath;
    if (isAdminUser)
    {
        filePath = SYSTEM_MESH_DIR;
    }
    else
    {
        filePath = USER_MESH_DIR;
    }
    filePath += MESHCONFIGURATIONFILEPATH;
    if ( !wxFileName::FileExists(filePath) )
    {
        return false;
    }

    wxXmlDocument doc;
    if ( !doc.Load(filePath) )
    {

        return false;
    }

    //get root element
    wxXmlNode* pRoot = doc.GetRoot();
    if (NULL == pRoot)
    {
        return false;
    }

    wxXmlNode* pChild = pRoot->GetChildren();
    bool bReturn = true;
    while ( NULL != pChild )
    {
        //read MeshLocalHostName element
        if ( _T("MeshLocalHostName") == pChild->GetName() )
        {
            wxString strLocalName;
            bReturn &= pChild->GetPropVal(_T("value"), &strLocalName);
            meshConfiguration.SetMeshLocalHostName(strLocalName);
        }
        //read MeshChannel element
        else if ( _T("MeshChannel") == pChild->GetName() )
        {
            wxString strMeshChannel;
            bReturn &= pChild->GetPropVal(_T("value"), &strMeshChannel);
            long lMeshChannel = 0;
            strMeshChannel.ToLong(&lMeshChannel);
            meshConfiguration.SetMeshChannel((int)lMeshChannel);
        }
        //read MeshAdvertiseTime element
        else if ( _T("MeshAdvertiseTime") == pChild->GetName() )
        {
            wxString strMeshAdvertiseTime;
            bReturn &= pChild->GetPropVal(_T("value"), &strMeshAdvertiseTime);
            long lMeshAdvertiseTime= 0;
            strMeshAdvertiseTime.ToLong(&lMeshAdvertiseTime);
            meshConfiguration.SetMeshAdvertiseTime((int)lMeshAdvertiseTime);
        }
        pChild = pChild->GetNext();
    }
    return bReturn;
}

