# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

"""
Various widgets used to display the results of a search.
"""

from elisa.core.utils.i18n import install_translation
from elisa.core.input_event import EventValue
from elisa.core.common import application

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.list_vertical import ListVertical
from elisa.plugins.pigment.widgets.list_horizontal import ListHorizontal
from elisa.plugins.pigment.widgets import notifying_list
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.graph.text import Text

from elisa.plugins.poblesec.widgets.sliced_image import SlicedImageHorizontal
from elisa.plugins.poblesec.widgets.loading_animation import LoadingAnimation

import pgm
from pgm.timing import implicit


_ = install_translation('poblesec')


class SearchResultNode(object):

    """
    A node in a list of search results for one given searcher.
    """

    def __init__(self, searcher, result_type, label, icon_resource, results):
        """
        Constructor.

        @param searcher:      the searcher entry point
        @type searcher:       C{str}
        @param result_type:   the type of results of this category
        @type result_type:    C{str}
        @param label:         the text label for this category of results
        @type label:          C{str}
        @param icon_resource: the icon resource for this category of results
        @type icon_resource:  C{str}
        @param results:       the list of results for this category
        @type results:        C{list} of L{elisa.plugins.base.models.*}
        """
        self.searcher = searcher
        self.result_type = result_type
        self.label = label
        self.icon_resource = icon_resource
        self.results = results


class SearchResultWidget(Widget):

    """
    A widget used to display the contents of a L{SearchResultNode}.
    """

    style_prefix = 'search-result'

    def __init__(self):
        super(SearchResultWidget, self).__init__()

        # Add a background drawable to receive events
        self._background = Image()
        self.add(self._background)
        self._background.bg_a = 0
        self._background.visible = True

        self.icon = Image()
        self.add(self.icon, forward_signals=False)
        self.icon.visible = True

        self.label = Text()
        self.add(self.label, forward_signals=False)
        self.label.visible = True

        self.total = Text()
        self.add(self.total, forward_signals=False)
        self.total.visible = True

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(SearchResultWidget, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == '%s-icon-x' % self.style_prefix:
                self.icon.x = value
            elif key == '%s-icon-y' % self.style_prefix:
                self.icon.y = value
            elif key == '%s-icon-height' % self.style_prefix:
                self.icon.height = value
            elif key == '%s-icon-alignment' % self.style_prefix:
                self.icon.alignment = eval('pgm.IMAGE_%s' % value.upper())
            elif key == '%s-icon-bga' % self.style_prefix:
                self.icon.bg_a = value
            elif key == '%s-label-x' % self.style_prefix:
                self.label.x = value
            elif key == '%s-label-y' % self.style_prefix:
                self.label.y = value
            elif key == '%s-label-height' % self.style_prefix:
                self.label.height = value
            elif key == '%s-label-ellipsize' % self.style_prefix:
                self.label.ellipsize = eval('pgm.TEXT_ELLIPSIZE_%s' % value.upper())
            elif key == '%s-label-bga' % self.style_prefix:
                self.label.bg_a = value
            elif key == '%s-label-fg-color' % self.style_prefix:
                self.label.fg_color = value
            elif key == '%s-total-x' % self.style_prefix:
                self.total.x = value
            elif key == '%s-total-y' % self.style_prefix:
                self.total.y = value
            elif key == '%s-total-width' % self.style_prefix:
                self.total.width = value
            elif key == '%s-total-height' % self.style_prefix:
                self.total.height = value
            elif key == '%s-total-alignment' % self.style_prefix:
                self.total.alignment = eval('pgm.TEXT_ALIGN_%s' % value.upper())
            elif key == '%s-total-bga' % self.style_prefix:
                self.total.bg_a = value
            elif key == '%s-total-fg-color' % self.style_prefix:
                self.total.fg_color = value


class SearchResultEntryNode(object):

    """
    A node in a list of search results that represents the results for one
    given searcher.
    """

    def __init__(self, searcher_entry, results):
        """
        Constructor.

        @param searcher_entry: a searcher entry
        @type searcher_entry:  L{elisa.plugins.poblesec.search_controller.SearcherEntry}
        @param results:        the result of a search
        @type results:         one of L{elisa.plugins.search.models.*}
        """
        self.searcher_entry = searcher_entry
        self.results = results


class SearchResultEntryWidget(Widget):

    """
    A widget used to display the contents of a L{SearchResultEntryNode}.

    @cvar item_widget_class: the class of the widget used to display one search
                             result node (category)
    @type item_widget_class: L{elisa.plugins.pigment.widgets.widget.Widget}
    """

    item_widget_class = None
    style_prefix = 'search-result-entry'

    def __init__(self):
        super(SearchResultEntryWidget, self).__init__()

        # Title
        self.icon = Image()
        self.add(self.icon)
        self.icon.visible = True

        self.label = Text()
        self.add(self.label)
        self.label.visible = True

        # FIXME: we need the frontend to get a reference to the gst_metadata
        # instance. This a cheap - UGLY - way to get the frontend without
        # changing a lot of client code. It is really ugly as we assume there
        # is only one frontend, which might not be the case in the future...
        self.frontend = application.interface_controller.frontends.values()[0]

        # Loading animation
        self.loading_animation = LoadingAnimation()
        self.add(self.loading_animation)
        self.loading_animation.visible = False

        # Vertical list of results
        self.list = ListVertical(widget_class=self.item_widget_class,
                                 visible_range_size=3)
        self.add(self.list)
        self.list.set_renderer(self._node_renderer)
        self.list.visible = True

        selector = SlicedImageHorizontal()
        theme = self.frontend.get_theme()
        selector_root = 'elisa.plugins.poblesec.selector'
        left_cap = theme.get_resource('%s_left_cap' % selector_root)
        right_cap = theme.get_resource('%s_right_cap' % selector_root)
        body = theme.get_resource('%s_body' % selector_root)
        selector.left_cap.set_from_file(left_cap)
        selector.right_cap.set_from_file(right_cap)
        selector.body.set_from_file(body)
        self.list.set_selector(selector)

        self._animated_self = implicit.AnimatedObject(self)
        self._animated_self.setup_next_animations(duration=200)

        self._clicked_id = self.list.connect('item-clicked',
                                             self._item_clicked_cb)

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(SearchResultEntryWidget, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == '%s-icon-x' % self.style_prefix:
                self.icon.x = value
            elif key == '%s-icon-y' % self.style_prefix:
                self.icon.y = value
            elif key == '%s-icon-width' % self.style_prefix:
                self.icon.width = value
            elif key == '%s-icon-height' % self.style_prefix:
                self.icon.height = value
            elif key == '%s-icon-layout' % self.style_prefix:
                self.icon.layout = eval('pgm.IMAGE_%s' % value.upper())
            elif key == '%s-icon-alignment' % self.style_prefix:
                self.icon.alignment = eval('pgm.IMAGE_%s' % value.upper())
            elif key == '%s-icon-bga' % self.style_prefix:
                self.icon.bg_a = value
            elif key == '%s-label-x' % self.style_prefix:
                self.label.x = value
            elif key == '%s-label-y' % self.style_prefix:
                self.label.y = value
            elif key == '%s-label-width' % self.style_prefix:
                self.label.width = value
            elif key == '%s-label-height' % self.style_prefix:
                self.label.height = value
            elif key == '%s-label-ellipsize' % self.style_prefix:
                self.label.ellipsize = eval('pgm.TEXT_ELLIPSIZE_%s' % value.upper())
            elif key == '%s-label-bga' % self.style_prefix:
                self.label.bg_a = value
            elif key == '%s-label-fg-color' % self.style_prefix:
                self.label.fg_color = value
            elif key == '%s-loading-x' % self.style_prefix:
                self.loading_animation.x = value
            elif key == '%s-loading-y' % self.style_prefix:
                self.loading_animation.y = value
            elif key == '%s-loading-width' % self.style_prefix:
                self.loading_animation.width = value
            elif key == '%s-loading-height' % self.style_prefix:
                self.loading_animation.height = value
            elif key == '%s-list-x' % self.style_prefix:
                self.list.x = value
            elif key == '%s-list-y' % self.style_prefix:
                self.list.y = value
            elif key == '%s-list-width' % self.style_prefix:
                self.list.width = value
            elif key == '%s-list-height' % self.style_prefix:
                self.list.height = value

    def clean(self):
        self.disconnect(self._clicked_id)
        super(SearchResultEntryWidget, self).clean()

    def update(self, searcher, result_model):
        """
        Update the list of results with the result model.

        This method should be implemented by subclasses depending on the type
        of search performed.

        @param searcher:     the searcher entry point
        @type searcher:      C{str}
        @param result_model: the results of a search
        @type result_model:  one of L{elisa.plugins.search.models.*}
        """
        raise NotImplementedError()

    def activate(self, item):
        """
        Activate one of the result categories.

        @param item: the activated category
        @type item:  L{SearchResultNode}
        """
        if len(item.results) == 0:
            return
        browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
        path = '/poblesec/search/results/%s/%s'
        path = path % (item.searcher, item.result_type)
        args = {item.result_type: item.results}
        deferred = browser.history.append_controller(path, item.label, **args)

    def handle_event(self, event_value):
        """
        Handle keyboard events.

        @param event_value: the value of the keyboard event
        @type event_value:  L{elisa.extern.enum.EnumValue}

        @return: C{True} if handling the event, C{False} otherwise
        @rtype:  C{bool}
        """
        if event_value in (EventValue.KEY_OK, EventValue.KEY_RETURN):
            self.activate(self.list.model[self.list.selected_item_index])
            return True
        elif event_value == EventValue.KEY_GO_DOWN:
            if self.list.selected_item_index < (len(self.list.model) - 1):
                self.list.selected_item_index += 1
                return True
        elif event_value == EventValue.KEY_GO_UP:
            if self.list.selected_item_index > 0:
                self.list.selected_item_index -= 1
                return True
        return False

    def _node_renderer(self, item, widget):
        self.frontend.load_from_theme(item.icon_resource, widget.icon)
        widget.label.label = item.label
        widget.total.label = str(len(item.results))

    def _item_clicked_cb(self, widget, item):
        self.activate(item)

    def do_focus(self, focus):
        if focus:
            # Transfer the focus to the list
            self.list.focus = True
            self._animated_self.opacity = 255
        else:
            self._animated_self.opacity = 100


class MusicSearchResultEntryWidget(SearchResultEntryWidget):

    """
    A widget that displays the search results for one given music searcher.
    """

    item_widget_class = SearchResultWidget

    def update(self, searcher, result_model):
        """
        @type result_model: L{elisa.plugins.search.models.MusicSearchResultModel}
        """
        res_root = 'elisa.plugins.poblesec.by'
        model = [SearchResultNode(searcher, 'artists', _('Artists'),
                                  '%s_artist' % res_root,
                                  result_model.artists),
                 SearchResultNode(searcher, 'albums', _('Albums'),
                                  '%s_album' % res_root,
                                  result_model.albums),
                 SearchResultNode(searcher, 'tracks', _('Tracks'),
                                  '%s_track' % res_root,
                                  result_model.tracks)]
        self.list.set_model(model)


class SearchResultsWidget(ListHorizontal):

    """
    A widget that displays the search results for all the searchers of a
    search.
    """

    # Currently the only supported media type is 'music'.
    # To be completed later on when we support other types of search.
    mediatype_to_widget = {'music': MusicSearchResultEntryWidget}

    def __init__(self, frontend, mediatype):
        """
        Constructor.

        @param frontend:  the frontend for which the widget is displayed
        @type frontend:   L{elisa.core.components.frontend.Frontend}
        @param mediatype: the type of media searched
        @type mediatype:  C{str}
        """
        su = super(SearchResultsWidget, self)
        su.__init__(widget_class=self.mediatype_to_widget[mediatype],
                    visible_range_size=2)
        self.frontend = frontend
        self.set_renderer(self._node_renderer)

        self._selected_id = self.connect('selected-item-changed',
                                         self._selected_item_changed_cb)

    def clean(self):
        self.disconnect(self._selected_id)
        super(SearchResultsWidget, self).clean()

    def set_searchers(self, searchers):
        """
        Set the list of searchers used for the search and for which results
        should be displayed.

        @param searchers: a list of searcher entries
        @type searchers:  C{list} of
                          L{elisa.plugins.poblesec.search_controller.SearcherEntry}
        """
        model = []
        for searcher_entry in searchers:
            model.append(SearchResultEntryNode(searcher_entry, None))
        self.set_model(notifying_list.List(model))

        if len(model) < self.visible_range_size:
            # FIXME: weird behaviour with the selectors and items' opacity when
            # changing the visible range size
            self.visible_range_size = len(model)

    def set_loading(self, searcher_entry, loading):
        """
        Start/stop the loading animation for one given searcher.

        @param searcher_entry: the searcher for which to give visual feedback
        @type searcher_entry:  L{elisa.plugins.poblesec.search_controller.SearcherEntry}
        @param loading:        C{True} to start the animation, C{False} to stop
                               it
        @type loading:         C{bool}
        """
        for index, item in enumerate(self.model):
            if item.searcher_entry == searcher_entry:
                break
        widget = self._widgets[self._widget_index_from_item_index(index)]
        widget.loading_animation.visible = loading
        widget.loading_animation.activate(loading)

    def update(self, searcher_entry, result_model):
        """
        Update the search results for a given searcher entry.

        @param searcher_entry: a searcher entry
        @type searcher_entry:  L{elisa.plugins.poblesec.search_controller.SearcherEntry}
        @param result_model:   the result of a new search
        @type result_model:    one of L{elisa.plugins.search.models.*}
        """
        for index, item in enumerate(self.model):
            if item.searcher_entry == searcher_entry:
                item.results = result_model
                self.model[index] = item
                break

    def handle_event(self, event_value):
        """
        Handle keyboard events.

        @param event_value: the value of the keyboard event
        @type event_value:  L{elisa.extern.enum.EnumValue}

        @return: C{True} if handling the event, C{False} otherwise
        @rtype:  C{bool}
        """
        if event_value == EventValue.KEY_GO_LEFT:
            if self.selected_item_index > 0:
                self.selected_item_index -= 1
            return True
        elif event_value == EventValue.KEY_GO_RIGHT:
            if self.selected_item_index < (len(self.model) - 1):
                self.selected_item_index += 1
            return True
        else:
            item_index = self.selected_item_index
            widget_index = self._widget_index_from_item_index(item_index)
            widget = self._widgets[widget_index]
            return widget.handle_event(event_value)

    def _node_renderer(self, item, widget):
        self.frontend.load_from_theme(item.searcher_entry.icon_resource,
                                      widget.icon)
        widget.label.label = item.searcher_entry.title
        widget.update(item.searcher_entry.searcher, item.results)

    def _selected_item_changed_cb(self, widget, item, prev_item):
        if item is not None:
            item_index = self.model.index(item)
            widget_index = self._widget_index_from_item_index(item_index)
            widget = self._widgets[widget_index]
            if self.focus or (self.focus_child is not None):
                widget.focus = True
        if prev_item is not None and item is not None:
            prev_item_index = self.model.index(prev_item)
            prev_widget_index = self._widget_index_from_item_index(prev_item_index)
            prev_widget = self._widgets[prev_widget_index]
            widget.list.selected_item_index = prev_widget.list.selected_item_index

    def _layout_widget(self, widget, position):
        # Private method overridden, do not call self.compute_opacity
        widget.position = (self.compute_x(position), 0.0, 0.0)

    def do_focus(self, focus):
        if focus:
            # Transfer the focus to the selected item's widget
            item_index = self.selected_item_index
            widget_index = self._widget_index_from_item_index(item_index)
            widget = self._widgets[widget_index]
            widget.focus = True
