# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.graph.group import Group
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.progressbar import ProgressBar
from elisa.plugins.pigment.widgets.const import *
from elisa.core.input_event import *
from elisa.core.media_uri import MediaUri
from elisa.core.utils.i18n import install_translation
import platform

try:
    from elisa.plugins.database.models import File as DBFile
except ImportError:
    # database missing
    DBFile = None

from elisa.core import common

_ = install_translation('poblesec')

from elisa.plugins.base.models.media import PlayableModel

from elisa.core.log import Loggable
import gobject

from twisted.internet import reactor, defer
from pgm.timing import implicit
import pgm, time
import gst
from gst import pbutils
import math
import os
import platform

class VisualisationError(Exception):
    pass

# The Visualisation Fake
class ElisaVisualisationBin(gst.Bin):

    def __init__(self, visualisation, size, pgm_image):
        """
        The ElisaVisualisationBin is a wrapper for the different visualisation
        elements of GStreamer.

        @param visualisation:   the GStreamer Element to use
        @type visualisation:    string
        @param size:            a size in pixel to calculate the visualisation
                                size for
        @type size:             int
        @param pgm_image:       the pgm_image that is used for sinking
        @type pgm_image:         L{pgm.graph.image.Image}
        """

        gst.Bin.__init__(self)

        self._visualisation = visualisation
        self._pgm_image = pgm_image

        self._capsfilter = gst.element_factory_make('capsfilter')
        self.add(self._capsfilter)

        caps_pad = self._capsfilter.get_pad('src')

        self._visu = gst.element_factory_make(self._visualisation)
        self.add(self._visu)

        height = int(round(size * pgm_image.absolute_height / pgm_image.absolute_width))
        height = (height + 3) & ~3

        visu_sink = self._visu.get_pad('sink')
        visu_src = self._visu.get_pad('src')

        caps = visu_src.get_pad_template().caps

        if caps == None:
            raise VisualisationError("The Visualisation element %s does not" \
                                     " have any Caps." % self._visualisation)

        caps = caps.copy()

        if caps.is_fixed() == False:
            caps.make_writable()
            for i in xrange(0, caps.get_size()):
                cur_struct = caps.get_structure(i)

                cur_struct.fixate_field_nearest_int('width', size)

                cur_struct.fixate_field_nearest_int('height', height)
        else:
            raise VisualisationError("The Caps of the GstElement are fixed.")

        self._capsfilter.set_property('caps', caps)

        self._ghost_sink = gst.GhostPad('sink', visu_sink )
        self._ghost_src = gst.GhostPad('src', caps_pad)

        self._visu.link(self._capsfilter)
        self.add_pad(self._ghost_sink)
        self.add_pad(self._ghost_src)


class PlayerOsdExit(Widget):

    def __init__(self):
        super(PlayerOsdExit, self).__init__()

        self.background = Image()
        self.add(self.background)
        self.background.bg_a = 0
        self.background.visible = True

class PlayerOsdStatus(Widget):
    """
    FIXME: regression regarding the 0.3.5 :

    The progress bar does not have borders
    The counter display only the same format
    The mouse is not supported
    The seeking with the mouse is not supported by the widget
    """

    style_prefix = 'osd-status'

    def __init__(self):
        super(PlayerOsdStatus, self).__init__()

        self.play_button = Image()
        self.add(self.play_button)
        self.play_button.bg_a = 0
        self.play_button.alignment = pgm.IMAGE_LEFT
        self.play_button.visible = False

        self.pause_button = Image()
        self.add(self.pause_button)
        self.pause_button.bg_a = 0
        self.pause_button.alignment = pgm.IMAGE_LEFT
        self.pause_button.visible = False
        
        self.seeking_dock_group = Group()
        self.add(self.seeking_dock_group)
        self.seeking_dock_group.visible = True
        
        self.dock = Image()
        self.seeking_dock_group.add(self.dock)
        self.dock.connect("pixbuf-loaded", self._body_image_loaded)
        self.dock.bg_a = 0
        self.dock.alignment = pgm.IMAGE_TOP_LEFT
        self.dock.visible = True

        self.text = Text()
        self.seeking_dock_group.add(self.text)
        self.text.bg_a = 0
        self.text.ellipsize = pgm.TEXT_ELLIPSIZE_END
        self.text.visible = True

        self.counter = Text()
        self.seeking_dock_group.add(self.counter)
        self.counter.bg_a = 0
        self.counter.alignment = pgm.TEXT_ALIGN_RIGHT
        self.counter.visible = True

        self.progress_bar = ProgressBar()
        self.seeking_dock_group.add(self.progress_bar)
        self.progress_bar.bg_a = 0
        self.progress_bar.visible = True
        self.progress_bar._orientation = HORIZONTAL

        self._update_style_properties(self._style.get_items())
        self.play()
        
    def _body_image_loaded(self, img):
        self.dock_image_loaded = True
        drawable_aspect_ratio = self.dock.absolute_width / \
                                            float(self.dock.absolute_height)
        image_aspect_ratio = self.dock.aspect_ratio[0] / \
                                            float(self.dock.aspect_ratio[1])

        #FIXME : does not work when Elisa has an aspect ratio < 1 need theme
        #update for this case.
        #previous_size = self.seeking_dock_group.size
        self.seeking_dock_group.width = image_aspect_ratio / \
                                                float(drawable_aspect_ratio)

    def _update_style_properties(self, props=None):
        super(PlayerOsdStatus, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == '%s-dock-x' % self.style_prefix:
                self.dock.x = value
            elif key == '%s-button-x' % self.style_prefix:
                self.pause_button.y = value
                self.play_button.y = value
            elif key == '%s-button-width' % self.style_prefix:
                self.pause_button.width = value
                self.play_button.width = value
            elif key == '%s-progress-bar-x' % self.style_prefix:
                self.progress_bar.x = value
            elif key == '%s-progress-bar-y' % self.style_prefix:
                self.progress_bar.y = value
            elif key == '%s-progress-bar-height' % self.style_prefix:
                self.progress_bar.height = value
            elif key == '%s-progress-bar-width' % self.style_prefix:
                self.progress_bar.width = value
            elif key == '%s-text-x' % self.style_prefix:
                self.text.x = value
            elif key == '%s-text-y' % self.style_prefix:
                self.text.y = value
            elif key == '%s-text-height' % self.style_prefix:
                self.text.height = value
            elif key == '%s-text-width' % self.style_prefix:
                self.text.width = value
            elif key == '%s-counter-x' % self.style_prefix:
                self.counter.x = value
            elif key == '%s-counter-y' % self.style_prefix:
                self.counter.y = value
            elif key == '%s-counter-height' % self.style_prefix:
                self.counter.height = value
            elif key == '%s-counter-width' % self.style_prefix:
                self.counter.width = value
            elif key == '%s-counter-visible' % self.style_prefix:
                if value == "True":
                    self.counter.visible = True
                else:
                    self.counter.visible = False

    def pause(self):
        self.play_button.visible = False
        self.pause_button.visible = True

    def play(self):
        self.pause_button.visible = False
        self.play_button.visible = True

class PlayerOsdStatusMixin(PlayerOsdStatus):
    __gsignals__ = {
        'progress-changed' : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,
                           (gobject.TYPE_INT,)),
        }
    factor = 100
    def __init__(self):
        self._formated_position = self._second_to_readable(-1)
        self._formated_duration = self._second_to_readable(-1)
        self._position = -1

        super(PlayerOsdStatusMixin, self).__init__()

        self.progress_bar.connect('index-changed', self._progress_bar_index_cb)

        self._update_counter()

    def set_duration(self, seconds):
        # hide the progress bar if we have no duration information (e.g: live stream)
        if seconds == -1:
            self.progress_bar.visible = False
        elif not self.progress_bar.visible:
            self.progress_bar.visible = True

        self.progress_bar.items_number = seconds * self.factor
        self._formated_duration = self._second_to_readable(seconds)
        self._update_counter()

    def set_position(self, seconds):
        if self._position == -1:
            # in the first call, we want to make the ui update
            self._formated_position = self._second_to_readable(seconds)

        f = seconds * self.factor
        self._position = f

        # update progress bar if it's visible, otherwise force an update of the
        # position label
        if self.progress_bar.visible:
            self.progress_bar.current_index = f
        else:
            self._formated_position = self._second_to_readable(seconds)
            self._update_counter()

    def seek_drag__get(self):
        return self.progress_bar.seek_drag

    seek_drag = property(seek_drag__get)

    def _second_to_readable(self, seconds):
        raise NotImplementedError()

    def _progress_bar_index_cb(self, widget, index, old_index):
        seconds = index / self.factor
        self._formated_position = self._second_to_readable(seconds)
        self._update_counter()

        if self._position == index:
            # it is because we setted the index
            return

        self.emit('progress-changed', seconds)

    def _update_counter(self):
        # no point in displaying a duration of 0 seconds
        if self._formated_duration != "00:00:00":
            counter = "%s / %s" % (self._formated_position, self._formated_duration)
        else:
            counter = str(self._formated_position)
        self.counter.label = counter

class VideoPlayerOsdStatus(PlayerOsdStatusMixin):

    def _second_to_readable(self, seconds):
        if seconds == -1:
            return "00:00:00"
        return time.strftime("%H:%M:%S", time.gmtime(seconds))


class PlayerOsdVolume(VideoPlayerOsdStatus):
    # FIXME: this inheritance is ugly because it has pause_button, play_button
    # and the play and pause method that are not related to this UI at all. A
    # more common mixin class would be very usefull
    style_prefix = 'osd-volume'
    
    def _progress_bar_index_cb(self, widget, index, old_index):
        if int(self._position) == index:
            # it is because we setted the index
            return
        
        seconds = float(index)
        
        self.emit('progress-changed', seconds)

class PlayerOsd(Widget):

    status_widget = VideoPlayerOsdStatus

    def __init__(self):
        super(PlayerOsd, self).__init__()

        self.status = self.status_widget()
        self.add(self.status)
        self.status.visible = True

        self.volume = PlayerOsdVolume()
        self.add(self.volume)
        self.volume.visible = True

        self.exit = PlayerOsdExit()
        self.add(self.exit)
        self.exit.visible = True

        #create animation stuff
        self.animated = implicit.AnimatedObject \
                                                (self)
        settings = {'duration': 350,
                    'transformation': implicit.DECELERATE}
        self.animated.setup_next_animations(**settings)
        self.animated.mode = implicit.REPLACE

        self._update_style_properties(self._style.get_items())

        self.opacity = 0

        self.is_visible = False
        self.time_before_hiding = 3.0
        self._hiding_dfr = None

    def _update_style_properties(self, props=None):
        super(PlayerOsd, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == 'osd-status-x':
                self.status.x = value
            elif key == 'osd-status-y':
                self.status.y = value
            elif key == 'osd-status-width':
                self.status.width = value
            elif key == 'osd-status-height':
                self.status.height = value
            elif key == 'time-before-hiding':
                self.time_before_hiding = value
            elif key == 'osd-volume-x':
                self.volume.x = value
            elif key == 'osd-volume-y':
                self.volume.y = value
            elif key == 'osd-volume-width':
                self.volume.width = value
            elif key == 'osd-volume-height':
                self.volume.height = value
            elif key == 'osd-exit-x':
                self.exit.x = value
            elif key == 'osd-exit-y':
                self.exit.y = value
            elif key == 'osd-exit-width':
                self.exit.width = value
            elif key == 'osd-exit-height':
                self.exit.height = value


    def hide(self):
        if self._hiding_dfr and self._hiding_dfr.active():
            self._hiding_dfr.cancel()
        self.is_visible = False
        self.status.progress_bar.sensitive = False
        self.volume.progress_bar.sensitive = False
        self.animated.opacity = 0

    def show(self):
        if self._hiding_dfr and self._hiding_dfr.active():
            self._hiding_dfr.cancel()
        self.is_visible = True
        self.status.progress_bar.sensitive = True
        self.volume.progress_bar.sensitive = True
        self.animated.opacity = 255
        if self.time_before_hiding:
            self._hiding_dfr = reactor.callLater(self.time_before_hiding, self.hide)

class Player(gobject.GObject, Loggable):

    STOPPED = 1
    PLAY = 2
    BUFFERING = 3
    PLAYING = 4
    PAUSED = 5

    __gsignals__ = {'status-changed':
                              (gobject.SIGNAL_RUN_LAST,
                              gobject.TYPE_BOOLEAN,
                              (gobject.TYPE_INT,)),
                    'playback-ended' :
                              (gobject.SIGNAL_RUN_LAST,
                              gobject.TYPE_NONE,
                              ()),
                    'player-missing-decoder':
                              (gobject.SIGNAL_RUN_LAST,
                               gobject.TYPE_NONE,
                               (gobject.TYPE_STRING, gobject.TYPE_STRING)),
                    'player-codec-error':
                              (gobject.SIGNAL_RUN_LAST,
                               gobject.TYPE_NONE,
                               ()),
                    'player-unknown-error':
                              (gobject.SIGNAL_RUN_LAST,
                               gobject.TYPE_NONE,
                               (gobject.TYPE_STRING,))
                              }

    def __init__(self):
        super(Player, self).__init__()
        # GStreamer pipeline setup

        # make sure typefindfunctions is loaded so we can remove the typefinders
        if platform.system() == 'Windows':
            gst.plugin_load_by_name('typefindfunctions')
            registry = gst.registry_get_default()
            typefinders = gst.type_find_factory_get_list()
            for typefinder in typefinders:
                if typefinder.get_name() in ('application/x-ape',
                    'application/x-apetag'):
                    registry.remove_feature(typefinder)

        self.pgm_sink = gst.element_factory_make('pgmimagesink')
        self.pipeline = gst.element_factory_make('playbin')
        self.pipeline.set_property('video-sink', self.pgm_sink)

        self.status = self.STOPPED
        self.last_error = None
        self.current_index = -1
        self.filename = None
        self.image = None
        self.volume_max = 2.0
        self.volume_increment = 0.10
        self.volume_decrement = 0.30
        self.seek_backward_seconds = 60
        self.seek_forward_seconds = 60
        self.key_repeat_count_second = 0.150
        self._last_key_time = time.time()
        self.playlist = []

        pbus = self.pipeline.get_bus()
        pbus.connect('message::eos', self._message_eos_cb)
        pbus.connect('message::error', self._message_error_cb)
        pbus.connect('message::state-changed', self._message_state_changed_cb)
        pbus.connect('message::buffering', self._message_buffering_cb)
        pbus.connect('message::element', self._message_element_cb)
        pbus.add_signal_watch()

        self._play_deferreds = []

    def _message_eos_cb(self, bus, message):
        self.pipeline.set_state(gst.STATE_NULL)
        self.play_next()

    def _message_buffering_cb(self, bus, message):
        self.status = self.BUFFERING
        self.emit('status-changed', self.status)

    def _message_error_cb(self, bus, message):
        err, msg = message.parse_error()
        self.last_error = (err, msg)
        self.warning("Gstreamer %s:%s" % (err, msg))
        self.pipeline.set_state(gst.STATE_NULL)
        code = message.structure['gerror'].code
        if code == gst.STREAM_ERROR_CODEC_NOT_FOUND:
            self.emit('player-codec-error')
        elif code != gst.STREAM_ERROR_FAILED:
            self.emit('player-unknown-error', err)

    def _message_state_changed_cb(self, bus, message):
        old_state, new_state, pending = message.parse_state_changed()

        if message.src != self.pipeline:
            return
        if new_state == gst.STATE_PLAYING:
            if self.status != self.PLAYING:
                self.status = self.PLAYING
                self.emit('status-changed', self.status)
        elif new_state == gst.STATE_PAUSED:
            if self.status != self.PAUSED:
                self.status = self.PAUSED
                self.emit('status-changed', self.status)
        elif self.status != self.STOPPED:
            self.status = self.STOPPED
            self.emit('status-changed', self.status)

    def _message_element_cb(self, bus, message):
        if pbutils.is_missing_plugin_message(message):
            caps =  message.structure['detail']
            details = caps.to_string().split(', ')
            mimetype = details[0]
            decoder = pbutils.get_decoder_description(mimetype)
            self.emit('player-missing-decoder', mimetype, decoder)

    def _load_subs(self, uri):
        found = False
        if uri.scheme == 'file':
            basename, ext = os.path.splitext(uri.filename)
            for sub_ext in ('srt', 'ssa', 'sub', 'txt'):
                sub_file = os.path.join(os.path.dirname(uri.path),
                                        "%s.%s" % (basename, sub_ext))
                if os.path.exists(sub_file):
                    sub_uri = MediaUri("file://%s" % sub_file)
                    self.pipeline.set_property('suburi', sub_uri)
                    self.pipeline.set_property('subtitle-font-desc', "Liberation Sans")
                    self.info("Loaded subtitles at %r", sub_uri)
                    found = True
                    break
        if not found:
            self.info("No subtitles found for %r", uri)
            self.pipeline.set_property('suburi', '')


    def set_drawable(self, value):
        self.image = value
        self.pgm_sink.set_property('image', value)

    def enqueue_to_playlist(self, playable_model):
        self.playlist.append(playable_model)

    def play_next(self):
        playlist_length = len(self.playlist)
        if self.current_index < playlist_length-1:
            self.play_at_index(self.current_index+1)
            return True
        else:
            self.emit('playback-ended')
            return False

    def play_previous(self):
        if self.current_index > 0 and len(self.playlist) > 0:
            self.play_at_index(self.current_index-1)
            return True
        return False

    def play_model(self, playable_model):
        # clear all the media that have never been played from the playlist
        self.playlist[self.current_index+1:] = []
        self.playlist.append(playable_model)
        self.play_at_index(len(self.playlist)-1)

    def _update_playcount_and_time(self, model):
        if model.uri.scheme != 'file':
            # database only support files
            return

        def update(result):
            if not result:
                return

            self.debug("updating playcount")
            result.playcount += 1
            result.last_played = time.time()

        store = common.application.store

        if not store or not DBFile:
            return

        dfr = store.get(DBFile, model.uri.path)
        dfr.addCallback(update)
        return dfr

    def play_at_index(self, index):
        if self.current_index == index:
            if self.status != self.PLAYING:
                self.play()
        else:
            for dfr in self._play_deferreds:
                if hasattr(dfr, 'cancel'):
                    dfr.cancel()

            def remove_from_play_deferreds(result, deferred):
                self._play_deferreds.remove(deferred)
                return result

            model = self.playlist[index]
            if not isinstance(model, PlayableModel):
                dfr = model.get_playable_model()
                self._play_deferreds.append(dfr)
                dfr.addBoth(remove_from_play_deferreds, dfr)
            else:
                dfr = defer.succeed(model)

            dfr.addCallback(self._play_playable_model_at_index, index)

    def _play_playable_model_at_index(self, model, index):
        self.filename = model.title or model.uri.filename
        self.stop()
        self.pipeline.set_property('uri', model.uri)
        self._load_subs(model.uri)

        # FIXME: make this configurable
        vis_plugin_name = 'libvisual_jess'
        if platform.system() == 'Windows':
            vis_plugin_name = 'goom2k1'

        try:
            visu = ElisaVisualisationBin(vis_plugin_name, 400,
                                         self.image)
            self.pipeline.set_property('vis-plugin', visu)
        except gst.ElementNotFoundError:
            pass

        self.play()
        self._update_playcount_and_time(model)
        self.current_index = index

    def stop(self):
        self.pipeline.set_state(gst.STATE_NULL)
        if self.image:
            self.image.clear()

    def pause(self):
        self.pipeline.set_state(gst.STATE_PAUSED)

    def play(self):
        current = self.pipeline.get_state(0)[1]
        if current != gst.STATE_PLAYING:
            if current != gst.STATE_PAUSED:
                self.status = self.BUFFERING
                self.emit('status-changed', self.status)
            self.pipeline.set_state(gst.STATE_PLAYING)

    def get_position(self):
        try:
            position, format = self.pipeline.query_position(gst.FORMAT_TIME)
            return position
        except gst.QueryError:
            return -1

    def set_position(self, value):
        if self.status not in (self.PLAYING, self.PAUSED):
            return

        event = self.pipeline.seek(1.0, gst.FORMAT_TIME,
                            gst.SEEK_FLAG_FLUSH | gst.SEEK_FLAG_KEY_UNIT,
                            gst.SEEK_TYPE_SET, value,
                            gst.SEEK_TYPE_SET, -1 )

        duration = self.get_duration()
        if duration != -1 and value >= duration:
            self.play_next()

    def get_duration(self):
        try:
            duration, format = self.pipeline.query_duration(gst.FORMAT_TIME)
            return duration
        except gst.QueryError:
            return -1

    def _can_handle_key(self):
        #FIXME Florian will not like that
        t = time.time()
        if (t - self._last_key_time) > self.key_repeat_count_second:
            self._last_key_time = t
            return True
        return False

    def get_volume(self):
        return self.pipeline.get_property('volume')

    def set_volume(self, volume):
        self.pipeline.set_property('volume', volume)

    def volume_up(self):
        volume = self.get_volume()
        if self._can_handle_key():
            volume += self.volume_increment
            if volume > self.volume_max:
                volume = self.volume_max
            self.set_volume(volume)
        return volume

    def volume_down(self):
        volume = self.get_volume()
        if self._can_handle_key():
            volume -= self.volume_decrement
            if volume < 0:
                volume = 0
            self.set_volume(volume)
        return volume

    def seek_backward(self):
        duration = self.get_duration()
        position = self.get_position()
        if position != -1 and duration > 0 and self._can_handle_key():
            position -= self.seek_backward_seconds * gst.SECOND
            if position < 0:
                # the user wants to seek to the previous track
                position = 0
                if self.play_previous():
                    return position
            self.set_position(position)
        return position

    def seek_forward(self):
        duration = self.get_duration()
        position = self.get_position()
        if position != -1 and duration > 0 and self._can_handle_key():
            position += self.seek_forward_seconds * gst.SECOND
            self.set_position(position)
        return position

class PlayerController(PigmentController):
    
    PlayerClass = Player

    def __init__(self):
        super(PlayerController, self).__init__()

        # FIXME: mute support should probably be moved to Player
        # volume before muting
        self.mute_volume = -1

        self._delayed_call = None

        self.background = Image()
        self.widget.add(self.background)
        self.background.bg_color = (0, 0, 0, 0)
        # FIXME: is that -1.0 necessary?
        self.background.z = -1.0
        self.background.visible = True

        self.player_osd = PlayerOsd()
        self.widget.add(self.player_osd)
        self.player_osd.bg_a = 0
        self.player_osd.opacity = 0
        self.player_osd.visible = True

        # Rotating for the loading animation
        self._interaction = None
        self._loading = False
        self._rotation_matrix = pgm.mat4x4_new_identity()
        self._rotation_matrix.translate(0.5, 0.5, 0.0)
        self._rotation_matrix.rotate_z(-math.pi / 30.0)
        self._rotation_matrix.translate(-0.5, -0.5, 0.0)
        self._create_interaction()

        self.player = self.PlayerClass()
        self.player.set_drawable(self.background)
        self.player.connect('status-changed',self._player_status_cb)

        self.player_osd.volume.set_duration(self.player.volume_max)
        self.player_osd.volume.set_position(self.player.get_volume())

        # connect to the click events
        self.player_osd.exit.background.connect \
                                ('clicked', self._exit_clicked_cb)

        self.player_osd.status.play_button.connect \
                                ('clicked', self._play_pause_clicked_cb)

        self.player_osd.status.pause_button.connect \
                                ('clicked', self._play_pause_clicked_cb)

        self.player_osd.status.connect \
                                ('progress-changed', self._progress_bar_cb)

        self.player_osd.volume.connect \
                                ('progress-changed', self._volume_bar_cb)
        self.player_osd.volume.play_button.connect \
                                ('clicked', self._volume_mute_cb)

        self.player_osd.volume.pause_button.connect \
                                ('clicked', self._volume_mute_cb)

        self._missing_decoders = set()
        self.player.connect('player-missing-decoder', self._player_missing_decoder_cb)
        self.player.connect('player-codec-error', self._check_missing_decoders_cb)

        self.player.connect('player-unknown-error', self._player_unknown_error_cb)

        self.player.connect('playback-ended', self._playback_ended)

    def _create_interaction(self):
        self._interaction = Image()
        self.widget.add(self._interaction)

        width, height = (0.5, 0.5)
        self._interaction.width, self._interaction.height = (width, height)
        self._interaction.x = width / 2.0
        self._interaction.y = height / 2.0

        # downscaling
        scale = pgm.mat4x4_new_identity()
        scale.translate(0.5, 0.5, 0.0)
        factor = 2.0
        scale.scale(factor, factor, 0.0)
        scale.translate(-0.5, -0.5, 0.0)

        self._interaction.bg_color = (0, 0, 0, 0)
        self._interaction.opacity = 255
        self._interaction.visible = True

    def _exit_clicked_cb(self, drawable, x, y, z, button, time, pressure):
        if not self.has_focus():
            return True

        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        self.player_osd.hide()
        main.hide_current_player()
        return True

    def _play_pause_clicked_cb(self, drawable, x, y, z, button, time, pressure):
        if not self.has_focus():
            return True

        self.toggle_play_pause()
        return True

    def _progress_bar_cb(self, widget, position):
        self.player.set_position(position * gst.SECOND)

    def _volume_bar_cb(self, widget, position):
        self.player_osd.volume.play()
        self.mute_volume = -1
        self.player.set_volume(position / float(self.player_osd.volume.factor))

    def _volume_mute_cb(self, drawable, x, y, z, button, time, pressure):
        if not self.has_focus():
            return True

        self.toggle_mute()
        return True

    def _player_status_cb(self, player, status):
        # show buffering even if focus is not set
        if status == player.BUFFERING:
            self._toggle_loading_animation(True)

        if not self.has_focus():
            return

        if self.player.filename == None:
            self.player_osd.status.text.label = ""
        else:
            self.player_osd.status.text.label = self.player.filename

        if status == player.PLAYING:
            self._toggle_loading_animation(False)
            self.player_osd.status.pause()
            self.player_osd.show()

            self._monitor()
            self.player_osd.status.set_duration \
                                        (self.player.get_duration()/gst.SECOND)

            self._check_missing_decoders_cb(self.player, partly_playable=True)
        elif status == player.PAUSED:
            self._stop_monitoring()
            self._toggle_loading_animation(False)
            self.player_osd.status.play()
        elif status == player.STOPPED:
            self._stop_monitoring()
            self._toggle_loading_animation(False)
            self.background.clear()
            self.player_osd.status.play()
        return True

    def _stop_monitoring(self):
        # stop the monitoring instantly
        if self._delayed_call and self._delayed_call.called == 0 and \
            self._delayed_call.cancelled == 0:
            self._delayed_call.cancel()

    def _monitor(self):
        if self._delayed_call and self._delayed_call.called == 0 and \
            self._delayed_call.cancelled == 0:
            # still a pending one
            return

        if not self.player_osd.status.seek_drag:
            # no update during drag seeking to not flicker around
            player_position = self.player.get_position()
            self.player_osd.status.set_position(player_position / gst.SECOND)

        self._delayed_call = reactor.callLater(0.10, self._monitor)

    def _player_missing_decoder_cb(self, player, mimetype, decoder):
        self._missing_decoders.add(decoder)

    def _check_missing_decoders_cb(self, player, partly_playable=False):
        if self._missing_decoders:
            missing_decoders = list(self._missing_decoders)
            self._missing_decoders.clear()
            main = self.frontend.retrieve_controllers('/poblesec')[0]
            if not partly_playable:
                self._toggle_loading_animation(False)
                self.player_osd.hide()
                main.hide_current_player()

            icon = 'elisa.plugins.poblesec.warning'
            title = _('Unable To Play Selected File')

            if partly_playable:
                text = _('Unfortunately Elisa can only partly play the selected file.')
            else:
                text = _('Unfortunately Elisa cannot play the selected file.')
            if missing_decoders:
                text += _(' Please download and install: ')
                # Localization of the names of the missing decoders is
                # automatically taken care of by gstreamer.
                text += _(', ').join(missing_decoders) + _('.')

            buttons = [(_('Close'), main.hide_popup)]

            main.show_popup(icon, title, text, buttons)

    def _player_unknown_error_cb(self, player, message):
        main = self.frontend.retrieve_controllers('/poblesec')[0]
        self._toggle_loading_animation(False)
        self.player_osd.hide()
        main.hide_current_player()

        icon = 'elisa.plugins.poblesec.warning'
        title = _('Unable To Play Selected File')

        text = _('Unfortunately Elisa cannot play the selected file.')
        text += _(' Gstreamer error: ') + message

        buttons = [(_('Close'), main.hide_popup)]

        main.show_popup(icon, title, text, buttons)

    def _playback_ended(self, player):
        main = self.frontend.retrieve_controllers('/poblesec')[0]
        self.player_osd.hide()
        self.background.clear()
        main.hide_current_player()

    def set_frontend(self, frontend):
        super(PlayerController, self).set_frontend(frontend)
        self._load_img_from_theme()

        self.frontend.viewport.connect('motion-notify-event', \
                                                        self._mouse_motion_cb)

    def _mouse_motion_cb(self, viewport, event):
         if self.has_focus():
             self.player_osd.show()

    def _load_img_from_theme(self):
        lft = self.frontend.load_from_theme
        theme = self.frontend.get_theme()
        
        osd_status = self.player_osd.status
        
        img_path = theme.get_resource('elisa.plugins.poblesec.osd_status_dock')
        osd_status.dock.set_from_file(img_path)
        
        lft('elisa.plugins.poblesec.osd_status_play_button',
                                      osd_status.play_button)
        lft('elisa.plugins.poblesec.osd_status_pause_button',
                                      osd_status.pause_button)

        osd_volume = self.player_osd.volume
        
        img_path = theme.get_resource('elisa.plugins.poblesec.osd_volume_dock')
        osd_volume.dock.set_from_file(img_path)
        
        lft('elisa.plugins.poblesec.osd_volume_button',
                                      osd_volume.play_button)

        lft('elisa.plugins.poblesec.osd_mute_button',
                                      osd_volume.pause_button)
        exit = self.player_osd.exit
        lft('elisa.plugins.poblesec.osd_exit_button',
                                      exit.background)

    def toggle_play_pause(self):
        if self.player.status == self.player.PLAYING:
            self.go_to_pause()
        else:
            self.go_to_play()

    def go_to_pause(self):
        self.player_osd.status.pause()
        if self.has_focus():
            self.player_osd.show()
        self.player.pause()

    def go_to_play(self):
        self.player_osd.status.play()
        if self.has_focus():
            self.player_osd.show()
        self.player.play()

    def toggle_mute(self):
        if self.mute_volume >= 0:
            self.player_osd.volume.play()
            self.player_osd.show()
            self.player.set_volume(self.mute_volume)
            self.mute_volume = -1
        else:
            self.player_osd.volume.pause()
            self.player_osd.show()
            self.mute_volume = self.player.get_volume()
            self.player.set_volume(0)

    def _toggle_loading_animation(self, enabled):
        if enabled and self.frontend:
            if self._loading:
                return
            wait_icon = self.frontend.get_theme().get_resource("elisa.plugins.poblesec.waiting_icon")
            self._interaction.visible = True
            self._interaction.set_from_file(wait_icon)
            self._interaction.set_name('wait icon')
            self._loading = True
            reactor.callLater(0.023, self._transform_mapping_matrix_cb)
        else:
            self._loading = False
            self._interaction.clear()
            self._interaction.visible = False

    def _transform_mapping_matrix_cb(self):
        if self._loading:
            self._interaction.mapping_matrix *= self._rotation_matrix
            reactor.callLater(0.017, self._transform_mapping_matrix_cb)

    def handle_input(self, manager, input_event):
        if input_event.value == EventValue.KEY_MENU:
            if self.player_osd.is_visible == False:
                self.player_osd.show()
                return True
            else:
                self.player_osd.hide()
                return False
        elif input_event.value == EventValue.KEY_OK or \
             input_event.value == EventValue.KEY_SPACE:
            self.toggle_play_pause()
            return True
        elif input_event.value == EventValue.KEY_PLAY:
            if self.player.status == self.player.PAUSED:
                self.go_to_play()
            return True
        elif input_event.value == EventValue.KEY_PAUSE:
            if self.player.status == self.player.PLAYING:
                self.go_to_pause()
            return True
        elif input_event.value == EventValue.KEY_GO_UP:
            self.player_osd.show()
            volume = self.player.volume_up()
            self.player_osd.volume.set_position(volume)
            return True
        elif input_event.value == EventValue.KEY_GO_DOWN:
            self.player_osd.show()
            volume = self.player.volume_down()
            self.player_osd.volume.set_position(volume)
            return True
        elif input_event.value == EventValue.KEY_GO_LEFT:
            self.player_osd.show()
            self.player.seek_backward()
        elif input_event.value == EventValue.KEY_GO_RIGHT:
            self.player_osd.show()
            self.player.seek_forward()
        elif input_event.value == EventValue.KEY_x or input_event.value == EventValue.KEY_PREVIOUS:
            self.player_osd.show()
            self.player.play_previous()
        elif input_event.value == EventValue.KEY_c or input_event.value == EventValue.KEY_NEXT:
            self.player_osd.show()
            self.player.play_next()
        elif input_event.value == EventValue.KEY_m:
            self.player_osd.show()
            self.toggle_mute()

        return False

