# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.const import *

from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.graph.text import Text
from pgm.timing.implicit import AnimatedObject

import pgm

import logging
import os, os.path


class Button(Widget):
    """A simple button with one image and/or one label.

    @ivar image: the path to the image to display
    @type image: string
    @ivar label: the button's label
    @type label: string
    @ivar focus_on_click: whether the button will grab focus when clicked
    @type focus_on_click: boolean
    """

    def __init__(self):
        """
        Add a background drawable to always get the events.
        """

        super(Button, self).__init__()

        self.focus_on_click = True

        # add a background drawble to receive events
        self._drawable = Image()
        self._drawable.bg_color = self.style.background_color
        self._drawable.visible = True
        self._animated_drawable = AnimatedObject(self._drawable)
        self.add(self._drawable)

        self._image_path = None
        self._image = None
        # no "image" is set, associate the animated image with the background
        # drawable
        self._animated_image = AnimatedObject(self._drawable)

        self._label = Text()
        self._label.font_family = "Bitstream DejaVu"
        self._label.font_height = 0.225
        self._label.fg_color = (255, 255, 255, 255)
        self._label.bg_color = (0, 0, 0, 0)
        self._label.ellipsize = pgm.TEXT_ELLIPSIZE_END
        self._label.visible = True

        self.add(self._label, forward_signals=False)
        self._label.x = 1/3.0
        self._label.y = 0.5

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(Button, self)._update_style_properties(props)

        if props is None:
            return

        if not self._animated_image:
            return

        for key, value in props.iteritems():
            if key == 'background-color':
                self._animated_drawable.bg_color = value
            # FIXME: dummy stuff
            elif key == 'padding':
                self._animated_image.x = value
            else:
                pass

    def image__get(self):
        """The path of the image"""
        return self._image_path

    def image__set(self, value):
        """
        Set the image at the specified path, creating the
        L{elisa.plugins.pigment.graph.image.Image} object, adding it to the
        scenegraph and associating an L{pgm.timing.implicit.Animated} object.

        @param value: the path of the image
        @type value: string
        """
        if self._image:
            self.remove(self._image)

        if value is None:
            self._image = None
            self._animated_image = None
            return

        assert(os.path.exists(value))

        self._image_path = value
        self._image = Image()
        self._image.set_from_file(value)
        self._image.bg_color = (0, 0, 0, 0)
        self._image.visible = True

        self._animated_image = AnimatedObject(self._image)
        self.add(self._image)

    image = property(image__get, image__set)

    def label__get(self):
        """Get the text of the label"""
        if self._label:
            return self._label.label

        return ''

    # FIXME: we actually don't know where to put the label in the button!
    def label__set(self, value):
        """Set the label text"""
        if self._label:
            self.remove(self._label)

        self._label = Text()
        self._label.label = value
        self._label.font_family = "Bitstream DejaVu"
        self._label.font_height = 0.225
        self._label.fg_color = (255, 255, 255, 255)
        self._label.bg_color = (0, 0, 0, 255)
        self._label.ellipsize = pgm.TEXT_ELLIPSIZE_END
        self._label.visible = True

        self._label.bg_color = (0, 0, 0, 0)
        self.add(self._label, forward_signals=False)
        self._label.x = 1/3.0
        self._label.y = 0.5

    label = property(label__get, label__set)

    def do_focus(self, value):
        if value:
            self.state = STATE_SELECTED
        else:
            self.state = STATE_NORMAL

    def do_clicked(self, x, y, z, button, time, pressure):
        logging.debug('Button clicked')

        if self.focus_on_click:
            self.focus = True
        return True

    def do_pressed(self, x, y, z, button, time, pressure):
        logging.debug('Button pressed')

        self.state = STATE_ACTIVE
        return True

    def do_released(self, x, y, z, button, time):
        logging.debug('Button released')

        if not self.focus_on_click:
            self.state = STATE_NORMAL
        return True

    def do_key_press_event(self, viewport, event, widget):
        if event.keyval == pgm.keysyms.Return:
            self.emit("clicked", 0, 0, 0, pgm.BUTTON_LEFT, 0, 0)
            self.state = STATE_ACTIVE
            if self.focus:
                self.state = STATE_SELECTED

    @classmethod
    def _demo_widget(cls, *args, **kwargs):
        from elisa.plugins.pigment.widgets.theme import Theme

        theme = Theme.get_default()

        widget = cls()
        img_file = theme.get_resource('elisa.plugins.pigment.widgets.yes')
        widget.image = img_file

        def on_click(widget, *args):
            print "ON CLICK", args

        widget.connect("clicked", on_click)
        widget.visible = True

        return widget


if __name__ == '__main__':
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    button = Button.demo()
    try:
        __IPYTHON__
    except NameError:
        pgm.main()

