# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2007-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Benjamin Kampmann <benjamin@fluendo.com>
#          Alessandro Decina <alessandro@fluendo.com>

from twisted.internet import reactor
from twisted.internet.error import ConnectionDone, ConnectionLost

from elisa.core.component import Component
from elisa.core.log import Loggable
from elisa.extern.log.log import getFailureMessage
from elisa.plugins.gstreamer.request import RequestQueue
from elisa.plugins.gstreamer.gst_metadata import able_to_handle, \
        supported_schemes, supported_keys
from elisa.plugins.amp.master import Master, SlaveProcessProtocol
from elisa.plugins.gstreamer.amp_protocol import GetMetadata, \
        GstMetadataMasterFactory

# use Loggable as the first base so that its debugging methods are used
class GstMetadataSlaveProcessProtocol(Loggable, SlaveProcessProtocol):
    """SlaveProcessProtocol that uses Loggable methods to log"""
    def __init__(self, master, slave_cookie):
        SlaveProcessProtocol.__init__(self, master, slave_cookie)
        Loggable.__init__(self)

    def __str__(self):
        return SlaveProcessProtocol.__str__(self)

class GstMetadataMaster(Loggable, Master):
    serverFactory = GstMetadataMasterFactory
    slaveProcessProtocolFactory = GstMetadataSlaveProcessProtocol 
    socket_prefix = 'elisa-metadata-'
    max_restart_retry = 10

    def __init__(self):
        Loggable.__init__(self)
        Master.__init__(self, 
                slave_runner='elisa.plugins.gstreamer.amp_slave.run_slave')
        self._requests = RequestQueue()
        self._restarted = 0
        self._next_request_call = None
        self._in_request = False
        self._process_next_request = None
        self._restart_defer = None
        self._last_slave_death_cause = None

    def _restart_callback(self, result):
        self._restart_defer = None
        self._restarted = 0
        self._last_slave_death_cause = None
        self._schedule_next_request()

    def _restart_errback(self, failure):
        self._restart_defer = None
        self._last_slave_death_cause = failure
        
        self.info('could not restart slave %s' % getFailureMessage(failure))
        self._try_restart_slave()
        return

    def get_metadata(self, metadata):
        dfr = self._requests.enqueue(metadata)
        if not self._in_request:
            self._schedule_next_request()

        return dfr

    def _schedule_next_request(self):
        if self._restart_defer or self._next_request_call:
            return

        assert self._next_request_call is None
        self._next_request_call = reactor.callLater(0, self._next_request)

    def _next_request(self):
        assert not self._in_request
        self._next_request_call = None

        try:
            slave = self._slaves.values()[0]
        except IndexError:
            self.warning('no slave running in _next_request')
            # no slave running, we try to restart one first
            self._try_restart_slave()
            return

        try:
            request = self._requests.dequeue()
        except IndexError:
            self.debug('metadata queue empty')
            return

        self.debug('starting request %s' % request.metadata)

        # FIXME: fix this properly but not so close to a release
        try:
            uri = str(request.metadata['uri'])
        except UnicodeError:
            if not request.cancelled:
                request.defer.errback()
            return

        keys = [{'key': key} for key in request.metadata.keys()
                if key != 'uri']

        dfr = slave.amp.callRemote(GetMetadata,  uri=uri, metadata=keys)
        self._in_request = True
        dfr.addCallback(self._get_metadata_cb, request)
        dfr.addErrback(self._get_metadata_eb, request)
    
    def _get_metadata_cb(self, result, request):
        self._in_request = False

        for dic in result['metadata']:
            request.metadata[dic['key']] = dic['value']

        self._schedule_next_request()
        request.defer.callback(request.metadata)

    def _restart_slave(self):
        self.warning('restarting slave')

        self._restart_defer = self.startSlaves(1)
        self._restart_defer.addCallback(self._restart_callback)
        self._restart_defer.addErrback(self._restart_errback)

    def _get_metadata_eb(self, failure, request):
        self._in_request = False

        self.debug('get metadata failed %s %s' %
                (request.metadata, getFailureMessage(failure)))

        if failure.check(ConnectionDone, ConnectionLost) and self._slaves_num:
            # FIXME: according to the Twisted Documentation we should not receive a
            # ConnectionLost in the case that the slave dies but it happens on
            # windows systems. For now we restart in any case but later we should
            # find out why it happens in the first place and fix it there.
            self._try_restart_slave()
        else:
            self._schedule_next_request()

        # errback this request so that we don't continue crashing on the same
        # file forever
        if not request.cancelled:
            request.defer.errback(failure)
    
    def _try_restart_slave(self):
        if self._restarted == self.max_restart_retry:
            self.warning('slave dead %s times, not trying anymore',
                    self._restarted)

            self._abort_pending_requests(self._last_slave_death_cause)
            return
        
        self.warning('restarting slave')
        self._restarted += 1        
        self._restart_defer = self.startSlaves(1)
        self._restart_defer.addCallback(self._restart_callback)
        self._restart_defer.addErrback(self._restart_errback)
    
    def _abort_pending_requests(self, failure):
        self.warning('aborting all requests, failure: %s' %
                    getFailureMessage(failure))
        while True:
            try:
                request = self._requests.dequeue()
            except IndexError:
                break

            if not request.cancelled:
                request.defer.errback(failure)


class GstMetadataAmpClient(Component):
    def initialize(self):
        def start_slaves_cb(result):
            return self
        
        self._master = GstMetadataMaster()
        self._master.start()
        dfr = self._master.startSlaves(1)
        dfr.addCallback(start_slaves_cb)

        return dfr

    def clean(self):
        def stop_slaves_cb(result):
            dfr = self._master.stop()
            dfr.addCallback(lambda result: self)

            return dfr

        dfr = self._master.stopSlaves()
        dfr.addCallback(stop_slaves_cb)

        return dfr

    def get_rank(self):
        return 10

    def able_to_handle(self, metadata):
        return able_to_handle(supported_schemes,
                supported_keys, metadata)

    def get_metadata(self, metadata):
        return self._master.get_metadata(metadata)
