# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Florian Boucault <florian@fluendo.com>

from elisa.plugins.poblesec.widgets.player.control_ribbon import Control
from elisa.plugins.pigment.widgets.const import *

from elisa.core.utils import defer
from elisa.core.utils.i18n import install_translation

from twisted.internet import reactor


_ = install_translation('poblesec')

class PlayPauseControl(Control):
    """
    Toggle the player state between play and pause.

    Its L{slave} passed to the constructor must be of type
    L{elisa.plugins.poblesec.player_video.PlayerController}.
    """

    def __init__(self, *args):
        super(PlayPauseControl, self).__init__(*args)
        # FIXME: should disconnect from the signal
        self.slave.player.connect('status-changed',
                                  self._on_player_status_changed)
        self._update_glyphs()
        self._update_caption()

    def _on_player_status_changed(self, player, status):
        self._update_glyphs()
        self._update_caption()

    def _update_glyphs(self):
        # import is done here to avoid circular import
        from elisa.plugins.poblesec.player_video import Player

        if self.slave.player.status is Player.PAUSED:
            self.glyphs = \
            {STATE_NORMAL: 'elisa.plugins.poblesec.player.glyphs.play_normal',
             STATE_SELECTED: 'elisa.plugins.poblesec.player.glyphs.play_selected',
             STATE_ACTIVE: 'elisa.plugins.poblesec.player.glyphs.play_active',
            }
        else:
            self.glyphs = \
            {STATE_NORMAL: 'elisa.plugins.poblesec.player.glyphs.pause_normal',
             STATE_SELECTED: 'elisa.plugins.poblesec.player.glyphs.pause_selected',
             STATE_ACTIVE: 'elisa.plugins.poblesec.player.glyphs.pause_active',
            }

    def _update_caption(self):
        # import is done here to avoid circular import
        from elisa.plugins.poblesec.player_video import Player

        if self.slave.player.status is Player.PAUSED:
            self.caption = _("Play")
        else:
            self.caption = _("Pause")

    def activate(self):
        self.slave.toggle_play_pause()
        return defer.succeed(None)

class StopControl(Control):
    """
    Stop the playback of the current media.

    Its L{slave} passed to the constructor must be of type
    L{elisa.plugins.poblesec.player_video.PlayerController}.
    """

    caption = _("Stop")
    glyphs = \
            {STATE_NORMAL: 'elisa.plugins.poblesec.player.glyphs.stop_normal',
             STATE_SELECTED: 'elisa.plugins.poblesec.player.glyphs.stop_selected',
             STATE_ACTIVE: 'elisa.plugins.poblesec.player.glyphs.stop_active',
            }

    def activate(self):
        self.slave.player.stop()
        self.slave.exit()
        return defer.succeed(None)

class SkipNextControl(Control):
    """
    Skip to the next track/video in the playlist and play it

    Its L{slave} passed to the constructor must be of type
    L{elisa.plugins.poblesec.player_video.PlayerController}.
    """

    caption = _("Fast Forward/Skip Next")
    glyphs = \
            {STATE_NORMAL: 'elisa.plugins.poblesec.player.glyphs.forward_normal',
             STATE_SELECTED: 'elisa.plugins.poblesec.player.glyphs.forward_selected',
             STATE_ACTIVE: 'elisa.plugins.poblesec.player.glyphs.forward_active',
            }

    def __init__(self, *args):
        super(SkipNextControl, self).__init__(*args)
        self._seek_call_later = None
        self._has_seeked = False
        self._seeking_interval = 0.2

    def activate(self):
        if not self._has_seeked:
            self.slave.player.play_next()
        return defer.succeed(None)

    def pre_activate(self):
        # no actual seek was performed yet
        self._has_seeked = False

        # schedule first seek event
        self._seek_call_later = reactor.callLater(self._seeking_interval,
                                                  self._seek_forwards)
        return defer.succeed(None)

    def de_pre_activate(self):
        # cancel regular seeking
        if self._seek_call_later != None and self._seek_call_later.active():
            self._seek_call_later.cancel()
        return defer.succeed(None)

    def _seek_forwards(self):
        # perform the actual seeking
        self._has_seeked = True

        if self.slave.player.seek_forward() < 0:
            # we don't want to seek again when seeking failed
            return

        # reschedule next seek event
        self._seek_call_later = reactor.callLater(self._seeking_interval,
                                                  self._seek_forwards)



class SkipPreviousControl(Control):
    """
    Skip to the previous track/video in the playlist and play it

    Its L{slave} passed to the constructor must be of type
    L{elisa.plugins.poblesec.player_video.PlayerController}.
    """

    caption = _("Rewind/Skip Previous")
    glyphs = \
            {STATE_NORMAL: 'elisa.plugins.poblesec.player.glyphs.rewind_normal',
             STATE_SELECTED: 'elisa.plugins.poblesec.player.glyphs.rewind_selected',
             STATE_ACTIVE: 'elisa.plugins.poblesec.player.glyphs.rewind_active',
            }

    def __init__(self, *args):
        super(SkipPreviousControl, self).__init__(*args)
        self._seek_call_later = None
        self._has_seeked = False
        self._seeking_interval = 0.2

    def activate(self):
        if not self._has_seeked:
            self.slave.player.play_previous()
        return defer.succeed(None)

    def pre_activate(self):
        # no actual seek was performed yet
        self._has_seeked = False

        # schedule first seek event
        self._seek_call_later = reactor.callLater(self._seeking_interval,
                                                  self._seek_backwards)
        return defer.succeed(None)

    def de_pre_activate(self):
        # cancel regular seeking
        if self._seek_call_later != None and self._seek_call_later.active():
            self._seek_call_later.cancel()
        return defer.succeed(None)

    def _seek_backwards(self):
        # perform the actual seeking
        self._has_seeked = True
        self.slave.player.seek_backward()

        # reschedule next seek event
        self._seek_call_later = reactor.callLater(self._seeking_interval,
                                                  self._seek_backwards)

class VolumeUpControl(Control):
    """
    Make the volume higher.

    Its L{slave} passed to the constructor must be of type
    L{elisa.plugins.poblesec.player_video.PlayerController}.
    """

    caption = _("Volume Up")
    glyphs = \
            {STATE_NORMAL: 'elisa.plugins.poblesec.player.volume.volup_normal',
             STATE_SELECTED: 'elisa.plugins.poblesec.player.volume.volup_selected',
             STATE_ACTIVE: 'elisa.plugins.poblesec.player.volume.volup_active',
            }

    def activate(self):
        self.slave.volume_up()
        return defer.succeed(None)

class VolumeDownControl(Control):
    """
    Make the volume lower.

    Its L{slave} passed to the constructor must be of type
    L{elisa.plugins.poblesec.player_video.PlayerController}.
    """

    caption = _("Volume Down")
    glyphs = \
            {STATE_NORMAL: 'elisa.plugins.poblesec.player.volume.voldown_normal',
             STATE_SELECTED: 'elisa.plugins.poblesec.player.volume.voldown_selected',
             STATE_ACTIVE: 'elisa.plugins.poblesec.player.volume.voldown_active',
            }

    def activate(self):
        self.slave.volume_down()
        return defer.succeed(None)
