# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.component import Component
from elisa.core.input_event import *

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.widgets import notifying_list
from elisa.plugins.poblesec.widgets.nothing_to_display import NothingToDisplay
from elisa.plugins.poblesec.actions import Action

from twisted.internet import defer


class ListController(PigmentController):

    def initialize(self):
        # FIXME: Tying this "nothing to display" widget to the list controller
        #        is a sacrilege, this should be refactored in a clean way.
        self.nothing_to_display_widget = NothingToDisplay()
        self.widget.add(self.nothing_to_display_widget)
        self.nothing_to_display_widget.visible = False
        dfr = super(ListController, self).initialize()
        return dfr

    def __init__(self):
        self._previous_clicked = None
        super(ListController, self).__init__()
        self.model = notifying_list.List()
        self.nodes = None


    def set_frontend(self, frontend):
        super(ListController, self).set_frontend(frontend)

        self.nodes_setup()

        self.nodes.connect('item-clicked', self._node_clicked_proxy)
        self.nodes.connect('selected-item-changed', self.node_selected)
        self.nodes.set_model(self.model)
        self.nodes.set_renderer(self.node_renderer)

        # by default the focus is forwarded to the list widget
        self.nodes.focus = self.widget.focus
        self.widget.connect('focus', self._on_focus)

    def stop_loading_animation(self):
        # FIXME: this uses some knowledge of the widgets that are used in the
        # controller: shouldn't be done like this. But now it allows to avoid
        # copy&paste code.
        try:
            self._previous_clicked.toggle_loading_animation(False)
        except AttributeError:
            pass

        self._previous_clicked = None

    def removed(self):
        self.stop_loading_animation()

    def sensitive_set(self, value):
        self._sensitive = value
        if not value:
            self.stop_loading_animation()

    def sensitive_get(self):
        return self._sensitive

    sensitive = property(fget=sensitive_get, fset=sensitive_set)

    def _on_focus(self, widget, focus):
        # always forward the focus to the list widget
        if focus:
            self.nodes.focus = focus

    def is_empty(self):
        if len(self.model) > 0:
            for item in self.model:
                if not isinstance(item, Action):
                    return False

        return True

    def toggle_empty_alert(self, value):
        if value:
            msg = getattr(self, 'empty_label', False)
            if not msg:
                return

            # remove the Actions from the model and from the screen
            self.model[:] = []
            self.actions[:] = []

            # FIXME: choose a better "generic" icon
            resource =  getattr(self, 'empty_icon', 'elisa.plugins.poblesec.file')
            self.frontend.load_from_theme(resource, self.nothing_to_display_widget.icon)
            self.nothing_to_display_widget.text.label = msg
            self.nothing_to_display_widget.visible = True
        else:
            self.nothing_to_display_widget.visible = False

    def nodes_setup(self):
        self.toggle_empty_alert(self.is_empty())

    def node_renderer(self, item, widget):
        pass

    def _node_clicked_proxy(self, widget, item):
        """
        This method is triggered by the widget item-clicked signal. It figures
        out the widget that was really clicked, checks if it is still in a
        previous_clicked mode or not sensitive to clicks. If we should react it
        calls self.node_clicked (the public method) with the selected widget as
        the first parameter and the item as the second parameter.

        # FIXME: as of today it seems no implementation is using the selected
        # widget so we should consider removing it from the node_clicked call
        """

        if not self.sensitive:
            return

        # FIXME: this is reversing the code that the list widget does to find
        # out the right item. We should refactor this part so that the we don't
        # have to calculate the index again.
        item_index = self.model.index(item)
        widget_index = self.nodes._widget_index_from_item_index(item_index)
        selected_widget = self.nodes._widgets[widget_index]

        if self._previous_clicked == selected_widget:
            # The user clicked again on the widget that was already clicked
            return

        try:
            # we try to activate the nice loading animation
            selected_widget.activate(previous=self._previous_clicked)
        except AttributeError:
            pass

        self._previous_clicked = selected_widget

        self.node_clicked(selected_widget, item)

    def node_clicked(self, node, item):
        pass

    def node_selected(self, widget, item, previous_item):
        pass

    def handle_input(self, manager, input_event):
        if input_event.value == EventValue.KEY_OK:
            try:
                item = self.nodes.model[self.nodes.selected_item_index]
                self._node_clicked_proxy(self.nodes, item)
            except IndexError:
                self.debug("selected-item-index out of range: %s" % self.nodes.selected_item_index)
            return True
        else:
            return super(ListController, self).handle_input(manager, input_event)


class GenericListViewMode(Component):

    """
    Generic view mode API.

    It defines a common API for clients. All one has to do is inherit from this
    class and implement the following methods:

     - C{get_label(item)}
     - C{get_sublabel(item)}
     - C{get_default_image(item)}
     - C{get_image(item, theme)}
     - C{get_preview_image(item, theme)}
    """

    def get_label(self, item):
        """
        Return a text to display in a label to represent an item.

        This call is asynchronous, it should return a
        L{elisa.core.utils.cancellable_defer.CancellableDeferred} that, when
        triggered, returns the text of the label.

        @param item: a list item
        @type item:  a subclass of L{elisa.core.components.model.Model}

        @return:     a cancellable deferred
        @rtype:      L{elisa.core.utils.cancellable_defer.CancellableDeferred}
        """
        return defer.fail(NotImplementedError())

    def get_sublabel(self, item):
        """
        Return a text to display in a sublabel to represent an item.

        This call is asynchronous, it should return a
        L{elisa.core.utils.cancellable_defer.CancellableDeferred} that, when
        triggered, returns the text of the sublabel.

        @param item: a list item
        @type item:  a subclass of L{elisa.core.components.model.Model}

        @return:     a cancellable deferred
        @rtype:      L{elisa.core.utils.cancellable_defer.CancellableDeferred}
        """
        return defer.fail(NotImplementedError())

    def get_default_image(self, item):
        """
        Return the path of a theme resource to display as a default image for
        an item.

        @param item:  a list item
        @type item:   a subclass of L{elisa.core.components.model.Model}

        @return:      the path of a theme resource to display as a default
                      image for the item
        @rtype:       C{str}
        """
        raise NotImplementedError()

    def get_image(self, item, theme):
        """
        Return the path to an image file to display as an image for an item.

        This call is asynchronous, it should return a
        L{elisa.core.utils.cancellable_defer.CancellableDeferred} that, when
        triggered, returns the path to an image file on disk (downloaded and
        cached if necessary).

        If no other image than the default one is necessary/available, this
        method should return C{None}.

        @param item:  a list item
        @type item:   a subclass of L{elisa.core.components.model.Model}
        @param theme: the frontend's current theme
        @type theme:  L{elisa.plugins.pigment.widgets.theme.Theme}

        @return:      a cancellable deferred or C{None}
        @rtype:       L{elisa.core.utils.cancellable_defer.CancellableDeferred}
        """
        return defer.fail(NotImplementedError())

    def get_preview_image(self, item, theme):
        """
        Return the path to an image file to display as a preview image for an
        item.

        This call is synchronous, if no preview image is available yet for the
        item or if no other image than the default one is necessary, it should
        return C{None}.

        @param item:  a list item
        @type item:   a subclass of L{elisa.core.components.model.Model}
        @param theme: the frontend's current theme
        @type theme:  L{elisa.plugins.pigment.widgets.theme.Theme}

        @return:      the path to an image file on disk or C{None}
        @rtype:       C{str} or C{None}
        """
        raise NotImplementedError()
