# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Florian Boucault <florian@fluendo.com>
#

from elisa.plugins.pigment.widgets.widget import Widget


class Square(Widget):
    """
    Widget that is constrained to be square, that is having an aspect ratio of
    1:1. One can change its size as usual by setting its width or height. It
    will automatically resize itself to stay visually square.
    
    Be careful, if the square is part of a group then its properties width and
    height might not be equal but still it will render visually square because
    these values are relative to the parent group.

    When both width and height are changed at the same time the height has
    priority over the width.
    """

    def __init__(self):
        super(Square, self).__init__()
        self._resized_id = self.connect('resized', self._resized_callback)

    def clean(self):
        super(Square, self).clean()
        self.disconnect(self._resized_id)

    def _square_height(self):
        if self._parent != None:
            height = self.width*self._parent.absolute_width/self._parent.absolute_height
            self.height = height
        else:
            self.height = self.width

    def _square_width(self):
        if self._parent != None:
            width = self.height*self._parent.absolute_height/self._parent.absolute_width
            self.width = width
        else:
            self.width = self.height

    def _resized_callback(self, notifier, previous_width, previous_height):
        if self._almost_equal(self.absolute_width, self.absolute_height):
            return

        if self._almost_equal(previous_height, self.absolute_height):
            self._square_height()
        else:
            self._square_width()

    def _almost_equal(self, float1, float2):
        return abs(float1-float2) <= 1e-6


if __name__ == '__main__':
    import pgm
    from elisa.plugins.pigment.graph.image import Image

    i = Image()
    i.visible = True

    s = Square.demo()
    s.add(i)

    g = Widget()
    g.add(s)
    g.visible = True

    try:
        __IPYTHON__
    except NameError:
        pgm.main()
