# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.const import *
from elisa.plugins.pigment.widgets import notifying_list

from elisa.plugins.pigment.graph.image import Image
from pgm.timing import implicit
from pgm.utils import maths

import pgm
import gobject

import math
import time as mod_time

from twisted.internet import reactor

class List(Widget):
    """
    Abstract list widget. It displays data stored in the L{model} list. Items
    of L{model} are rendered into widgets instances of the widget class passed
    to the constructor using the specified L{renderer} function.

    Usage example for a list of strings rendered in Text drawables::

        l = List(Text)

        def renderer(item, widget):
            widget.label = str(item)
            widget.bg_a = 0

        model = range(100)

        l.set_renderer(renderer)
        l.set_model(model)


    Emit the signals:
      - item-clicked: when an item of the list is clicked
      - selected-item-changed: when the selected item in the list changes

    @ivar selected_item_index:    index of the currently selected item in the
                                  L{model}
    @type selected_item_index:    C{int}
    @ivar visible_range_size:     number of rendered items
    @type visible_range_size:     C{float}
    @ivar model:                  list of data items that is rendered by the
                                  list widget
    @type model:                  C{list}
    @ivar drag_motion_resolution: minimum time between 2 drag events in milliseconds
    @type drag_motion_resolution: C{int}
    @ivar drag_threshold:         amount of movement needed to activate dragging
                                  in canvas coordinates
    @type drag_threshold:         C{float}
    @ivar animated:               C{True} if the list is animated, C{False} otherwise
    @type animated:               C{bool}
    @ivar preloaded:              number of items that are prerendered before
                                  they become visible
    @type preloaded:              C{int}
    @ivar focus_on_click:         whether the button will grab focus when clicked
    @type focus_on_click:         C{bool}
    """

    __gsignals__ = {
        'item-clicked': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                (gobject.TYPE_PYOBJECT,)),
        'selected-item-changed': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,
                (gobject.TYPE_PYOBJECT, gobject.TYPE_PYOBJECT)),
    }

    model = []
    drag_motion_resolution = 15
    drag_threshold = 0.05
    animated = True
    preloaded = 5
    widget_signals = {'clicked': 'item-clicked'}

    def __init__(self, widget_class, visible_range_size=7):
        """
        @param widget_class: widget type used to render the items of the model
        @type widget_class: type inheriting from
                            L{elisa.plugins.pigment.widgets.widget.Widget}
        @param visible_range_size: DOCME
        @type visible_range_size:  int
        """
        super(List, self).__init__()
        self._render_all_call = None

        self.focus_on_click = True

        # widget instances used for rendering the model data
        self._widgets = []
        self._widget_class = widget_class
        self._renderer = None

        self._selected_item_index = -1
        self._visible_range_start = 0.0

        # selector
        self._selector = None
        self._animated_selector = None

        self._last_drag_motion = 0
        self.visible_range_size = visible_range_size

        # signals support and mouse dragging support
        self._drag_zone = Image()
        self.add(self._drag_zone, forward_signals=True)
        self._drag_zone.bg_color = (0, 0, 0, 0)
        self._drag_zone.visible = True

        # current dragging state
        self._dragging = False
        self._drag_accum = 0

        # animation support
        attributes = ("visible_range_start",)
        self._animated = implicit.AnimatedObject(self, attributes)
        self._animated.setup_next_animations(duration=500,
                                             transformation=implicit.DECELERATE)
        self._animated.mode = implicit.REPLACE

        self._update_style_properties(self._style.get_items())

    # Public API

    def set_model(self, model):
        """
        Set the list of items that are rendered by the list widget. Further
        updates to the model are automatically picked up by the list widget.

        @param model: list of items that should be rendered by the list widget
        @type model: list
        """
        self.model = model

        if isinstance(self.model, notifying_list.List):
            # FIXME: it should save the connection ids in order to disconnect later
            self.model.notifier.connect('items-changed', self._on_items_changed)
            self.model.notifier.connect('items-deleted', self._on_items_deleted)
            self.model.notifier.connect('items-inserted', self._on_items_inserted)
            self.model.notifier.connect('items-reordered', self._on_items_reordered)

        self._render_all()
        self._set_selector_visibility()

        # no item from self.model was selected, select the first one by default
        if self.selected_item_index == -1 and len(self.model) > 0:
            self.selected_item_index = 0

    def set_renderer(self, renderer):
        """
        Function used by the list widget to render an item of the model. It
        takes an item of the model as the first argument and an instance of
        the widget class passed to the constructor of the list widget as the
        second argument.

        @param renderer: function called to render an item of the model to the
                         widget used for rendering
        @type renderer: callable
        """
        self._renderer = renderer
        self._render_all()

    def set_selector(self, selector):
        if self._selector:
            self._selector.clean()

        self._selector = selector

        self._animated_selector = implicit.AnimatedObject(self._selector)
        settings = {'duration': 200,
                    'transformation': implicit.DECELERATE}
        self._animated_selector.setup_next_animations(**settings)
        self._animated_selector.mode = implicit.REPLACE

        self.add(self._selector)

        self._prepare_selector()
        self._layout_selector()
        self._set_selector_visibility()

        if self.focus:
            self._selector.opacity = 255
        else:
            self._selector.opacity = 0

    # Implementation methods

    def _connect_widget(self, widget):
        for widget_signal, local_signal in self.widget_signals.iteritems():
            self._connect_widget_signal_to_own_with_model(widget,
                    widget_signal, local_signal)

    def _connect_widget_signal_to_own_with_model(self, widget, widget_signal,
                                                 local_signal):
        """
        @param widget: the widget to connect to
        @type widget:  elisa.plugins.pigment.widgets.widget.Widget
        @param widget_signal: the name of the signal of the widget
        @type widget_signal:  str
        @param local_signal: the name of the signal of the list
        @type local_signal:  str
        """
        widget.connect(widget_signal, self._emit_item_clicked, local_signal)

    def _emit_item_clicked(self, widget, *args):
        name = args[-1]
        index = self._widgets.index(widget)
        real_index = self._item_index_from_widget_index(index)
        model = self.model[real_index]

        self.emit(name, model)
        return True

    def _on_items_changed(self, notifier, index, items):
        widget_index = self._widget_index_from_item_index(index)
        start = max(widget_index, 0)
        end = min(widget_index+len(items), len(self._widgets))
        self._render_widgets(start, end)

    def _on_items_deleted(self, notifier, index, items):
        # update self.selected_item_index so that it is always corresponding
        # to an actual item
        if self.selected_item_index >= len(self.model):
            self.selected_item_index = len(self.model)-1

        widget_index = self._widget_index_from_item_index(index)
        start = max(widget_index, 0)
        end = len(self._widgets)
        self._render_widgets(start, end)

        self._set_selector_visibility()

    def _on_items_inserted(self, notifier, index, items):
        if len(items) == len(self.model):
            # case where the list was empty before that insertion
            # check that self.selected_item_index corresponds to an actual
            # item
            start = self._selected_to_range_start(self.selected_item_index)
            self.visible_range_start = start

            if self.selected_item_index == -1:
                # no item from self.model was selected, select the first one by default
                self.selected_item_index = 0
            elif self.selected_item_index >= len(self.model):
                self.selected_item_index = len(self.model)-1
        else:
            # update self.selected_item_index if the inserted items were before the
            # selected one so that the selected item stays selected
            if index <= self.selected_item_index and len(self.model) > 1:
                self.selected_item_index += len(items)

        widget_index = self._widget_index_from_item_index(index)
        start = max(widget_index, 0)
        end = len(self._widgets)
        self._render_widgets(start, end)

        self._set_selector_visibility()

    def _on_items_reordered(self, notifier):
        self._render_all()

    def _move_widgets_from_beginning_to_end(self, number):
        moved = self._widgets[:number]
        self._widgets[:number] = []
        self._widgets[len(self._widgets):] = moved

    def _move_widgets_from_end_to_beginning(self, number):
        moved = self._widgets[-number:]
        self._widgets[-number:] = []
        self._widgets[:0] = moved

    def _item_index_from_widget_index(self, widget_index):
        item_index = widget_index+int(self._visible_range_start)-self.preloaded
        return item_index

    def _widget_index_from_item_index(self, item_index):
        widget_index = item_index-int(self._visible_range_start)+self.preloaded
        return widget_index

    def _render_widgets(self, start, end):
        if self._renderer is None:
            return

        for i, widget in enumerate(self._widgets[start:end+1]):
            item_index = self._item_index_from_widget_index(i+start)
            if item_index >= 0 and item_index < len(self.model):
                item = self.model[item_index]
                gobject.idle_add(self._renderer, item, widget)
                widget.visible = True
            else:
                widget.visible = False

    def _render_all(self):
        if self._render_all_call is not None:
            return

        def render():
            self._render_all_call = None
            self._render_widgets(0, len(self._widgets)-1)

        self._render_all_call = reactor.callLater(0, render)

    def visible_range_start__get(self):
        return self._visible_range_start

    def visible_range_start__set(self, visible_range_start):
        old_start = self._visible_range_start
        self._visible_range_start = visible_range_start
        delta = int(math.modf(visible_range_start)[1]-math.modf(old_start)[1])

        if abs(delta) >= len(self._widgets):
            self._render_all()
        elif delta >= 1:
            self._move_widgets_from_beginning_to_end(delta)
            self._render_widgets(len(self._widgets)-delta,
                                 len(self._widgets)-1)
        elif delta <= -1:
            self._move_widgets_from_end_to_beginning(-delta)
            self._render_widgets(0, -delta-1)

        self._layout_all_widgets()

    visible_range_start = property(visible_range_start__get, visible_range_start__set)

    def visible_range_size__get(self):
        return self._visible_range_size

    def visible_range_size__set(self, visible_range_size):
        assert (visible_range_size != 0)
        self._visible_range_size = visible_range_size

        # FIXME: this is very suboptimal, we should not re-create all the
        # widgets, only those that are needed
        self._create_widgets()

        # set visible range start accordingly
        self._visible_range_start = self._selected_to_range_start(self._selected_item_index)

        # FIXME: this is very suboptimal, we should not re-render all the
        # widgets, only those that are needed
        self._render_all()

        self._prepare_all_widgets()
        self._layout_all_widgets()
        self._prepare_selector()
        self._layout_selector()

    visible_range_size = property(visible_range_size__get, visible_range_size__set)

    def _create_widgets(self):
         # FIXME: do not destroy and reload everything; support floats
        for widget in self._widgets:
            widget.clean()
        self._widgets[:] = []

        nb_widgets = self._visible_range_size+2*self.preloaded
        for i in xrange(nb_widgets):
            widget = self._widget_class()
            self._widgets.append(widget)
            self.add(widget, forward_signals=False)
            self._prepare_widget(widget)
            self._connect_widget(widget)

    def selected_item_index__get(self):
        return self._selected_item_index

    def selected_item_index__set(self, index):
        self._stop_deceleration()
        if len(self.model) == 0:
            # if there is no data let any positive index so that it is
            # possible to preselect an item that is planned to be added
            index = max(index, 0)
        else:
            index = maths.clamp(index, 0, len(self.model)-1)
        index = int(round(index))

        visible_range_start = self._selected_to_range_start(index)

        if self.animated and len(self.model) > 0:
            self._animated.visible_range_start = visible_range_start
        else:
            if self.animated:
                self._animated.stop_animations()
            self.visible_range_start = visible_range_start

        prev_selected = self._selected_item_index
        self._selected_item_index = index
        if prev_selected != index:
            try:
                item = self.model[index]
            except IndexError:
                item = None
            try:
                prev_item = self.model[prev_selected]
            except IndexError:
                prev_item = None
            gobject.idle_add(self.emit, 'selected-item-changed', item, prev_item)

    selected_item_index = property(selected_item_index__get, selected_item_index__set)

    def _selected_to_range_start(self, selected):
        half_size = (self.visible_range_size-1.0)/2.0
        if selected <= half_size or len(self.model) <= self.visible_range_size:
            visible_range_start = 0.0
        elif selected >= len(self.model)-half_size:
            visible_range_start = len(self.model)-self.visible_range_size
        else:
            visible_range_start = selected-half_size
        return visible_range_start

    def _range_start_to_selected(self, range_start):
        half_size = (self.visible_range_size-1.0)/2.0
        selected = range_start + half_size
        selected = int(round(selected))
        selected = maths.clamp(selected, 0, len(self.model)-1)
        return selected

    def _layout_widget(self, widget, position):
        raise NotImplementedError

    def _prepare_all_widgets(self):
        for widget in self._widgets:
            self._prepare_widget(widget)

    def _prepare_widget(self, widget):
        pass

    def _prepare_selector(self):
        pass

    def _layout_all_widgets(self):
        if not self.absolute_visible:
            return

        # we assume that the widgets are created and loaded with their
        # corresponding data from the model
        delta, start_item = math.modf(self._visible_range_start)

        for i, widget in enumerate(self._widgets):
            position = -delta + i - self.preloaded
            self._layout_widget(widget, position)

    def _layout_selector(self):
        if not self._selector:
            return

        visible_range_start = self._selected_to_range_start(self.selected_item_index)
        selected_item_index = self.selected_item_index - visible_range_start

        if self._selector.visible and self._selector.opacity != 0:
            selector = self._animated_selector
        else:
            selector = self._selector
            selector.z = 1.0

        self._actual_layout_selector(selector, selected_item_index)

    def _set_selector_opacity(self):
        if self._selector is None:
            return

        if self.focus:
            self._animated_selector.opacity = 255
        else:
            self._animated_selector.opacity = 0

    def _set_selector_visibility(self):
        if not self._selector:
            return

        # FIXME: visibility of the selector should be dynamic depending on how
        # the content of self.model evolves
        if len(self.model) == 0:
            self._selector.visible = False
        else:
            self._selector.visible = True

    def compute_height(self, index):
        raise NotImplementedError

    def compute_width(self, index):
        raise NotImplementedError

    def compute_x(self, index):
        raise NotImplementedError

    def compute_y(self, index):
        raise NotImplementedError

    def compute_z(self, index):
        return 0.0

    def compute_opacity(self, index):
        full_opacity = 255
        shaded_opacity = 100
        invisible_opacity = -80

        # make the transformation symmetrical
        if index <= self._visible_range_size/2.0:
            # beginning of the visible items case
            index = index
            start = self._visible_range_start
        else:
            # end of the visible items case
            index = self._visible_range_size - index - 1
            if len(self.model) >= self._visible_range_size:
                start = len(self.model) - self._visible_range_start - \
                        self._visible_range_size
            else:
                start = 1.0

        if start <= 0 or index >= 1 or self._visible_range_size <= 1.0:
            opacity = full_opacity
        elif index >= 0:
            opacity = maths.lerp(shaded_opacity, full_opacity, index)
        elif index < 0:
            if start >= 1:
                opacity = maths.lerp(invisible_opacity, shaded_opacity, index+1)
            else:
                opacity = maths.lerp(invisible_opacity, full_opacity, index+1)

        opacity = max(min(255, opacity), 0)

        return opacity


    # Overridden from Group

    def visible__set(self, value):
        old_value = self.absolute_visible
        super(List, self).visible__set(value)
        if self.absolute_visible and not old_value:
            self._layout_all_widgets()

    visible = property(Widget.visible__get, visible__set)

    # Signals support methods
    def do_focus(self, focus):
        self._set_selector_opacity()

    def do_released(self, x, y, z, button, time):
        if not self.focus_on_click:
            self.state = STATE_NORMAL
        return True

    def do_scrolled(self, x, y, z, direction, time):
        if direction == pgm.SCROLL_UP:
            if self._selected_item_index > 0:
                self.selected_item_index -= 1
        else:
            if self._selected_item_index < (len(self.model) - 1):
                self.selected_item_index += 1
        return True

    def do_drag_begin(self, x, y, z, button, time, pressure):
        if self._selector != None:
            self._animated_selector.opacity = 0

        self._dragging = True

        self._initial = (x, y, time)
        self._animated.stop_animations()
        self._stop_deceleration()

        self._drag_accum = 0

        self.speed = 0.0
        return True

    def do_drag_end(self, x, y, z, button, time):
        self._dragging = False

        if self._drag_accum > self.drag_threshold:
            self._current_time = mod_time.time()
            self._deceleration_source = gobject.timeout_add(17, self._decelerate)

        self._set_selector_opacity()

        return True

    def do_item_clicked(self, item):
        if self.focus_on_click:
            self.focus = True

        self.selected_item_index = self.model.index(item)

    def do_selected_item_changed(self, item, prev_item):
        self._layout_selector()

    # Inertia support methods
    # FIXME: they should be part of the animation framework

    deceleration = 8.0
    def _decelerate(self):
        self._previous_time = self._current_time
        self._current_time = mod_time.time()
        delta = self._current_time - self._previous_time

        if self.speed > 0.0:
            self.speed -= self.deceleration*delta
            if self.speed > 0.0:
                self.visible_range_start -= self.speed*delta

                # block the movement if it reaches the first item
                if self._range_start_to_selected(self._visible_range_start) > 0:
                    return True

        elif self.speed < 0.0:
            self.speed += self.deceleration*delta
            if self.speed < 0.0:
                self.visible_range_start -= self.speed*delta

                # block the movement if it reaches the last item
                if self._range_start_to_selected(self._visible_range_start) < len(self.model)-1:
                    return True

        self.selected_item_index = self._range_start_to_selected(self._visible_range_start)

        return False

    def _stop_deceleration(self):
        try:
            gobject.source_remove(self._deceleration_source)
        except AttributeError:
            pass

    @classmethod
    def _demo_widget(cls, *args, **kwargs):
        from elisa.plugins.pigment.graph.text import Text
        from elisa.plugins.pigment.graph.image import Image
        from elisa.plugins.pigment.widgets.notifying_list import List
        import random

        widget = cls(Text)
        widget.visible = True

        model = List(range(100))
        def renderer(item, widget):
            widget.label = str(item)
            widget.alignment = pgm.TEXT_ALIGN_CENTER
            widget.bg_a = 0

        widget.set_renderer(renderer)
        widget.set_model(model)

        widget.position = (0.5, 0.5, 0.0)
        widget.size = (3.0, 0.3)

        selector = Image()
        selector.bg_color = (255, 0, 0, 100)
        widget.set_selector(selector)

        def item_clicked_cb(self, item):
            index = self.model.index(item)

        def key_press_event_cb(self, viewport, event, grid):
            if event.keyval == pgm.keysyms.a:
                self.model.insert(self.selected_item_index+1, "test" + str(random.randint(0, 10000)))
            elif event.keyval == pgm.keysyms.d:
                item = self.model[self.selected_item_index]
                self.model.remove(item)

        widget.connect('item-clicked', item_clicked_cb)
        widget.connect('key-press-event', key_press_event_cb)

        return widget

    def clean(self):
        for widget in self._widgets:
            widget.clean()
        self._widgets=[]
        self._animated.stop_animations()
        self._animated = None
        self.animated = False

        if self._selector is not None:
            self._selector.clean()
            self._selector = None
        if self._animated_selector is not None:
            self._animated_selector.stop_animations()
            self._animated_selector = None

        self._drag_zone.clean()
        self._drag_zone = None

        if self._render_all_call is not None:
            self._render_all_call.cancel()
            self._render_all_call = None

        return super(List, self).clean()

