# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author:  Jess Corrius <jcorrius@fluendo.com>

from elisa.core import common
from elisa.core.media_uri import MediaUri
from elisa.core.utils.i18n import install_translation
from elisa.core.metadata_manager import IncompleteMetadataResponse

from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list import GenericListViewMode
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController, DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController
from elisa.plugins.poblesec.link import Link

from elisa.plugins.poblesec.actions import Action
from elisa.plugins.database.actions import \
    VideoPlayAllAction, VideoShuffleAction, VideoAddToFavoritesAction

from elisa.plugins.favorites.models import FavoritesItem

from elisa.plugins.base.models.media import PlayableModel

from elisa.plugins.database.models import Video, File

from twisted.internet import defer
from twisted.internet import task

import os, datetime
from random import shuffle

_ = install_translation('database')

def video_lib_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/video_library'
    link.label = _('Video Library')
    link.icon = 'elisa.plugins.poblesec.video_library'
    controller.model.append(link)

    return defer.succeed(None)

def video_lib_all_videos_decorator(controller):
    tracks = Link()
    tracks.controller_path = '/poblesec/database/video/list'
    tracks.label = _('All videos')
    tracks.icon = 'elisa.plugins.poblesec.file_video'
    controller.model.append(tracks)
    return defer.succeed(None)
    
def video_lib_date_decorator(controller):
    times = Link()
    times.controller_path = '/poblesec/database/video/by_date'
    times.label = _('Browse by date')
    times.icon = 'elisa.plugins.poblesec.by_decade'
    controller.model.append(times)
    return defer.succeed(None)

class ThumbnailMixin(object):

    def __init__(self):
        # FIXME: we need the frontend to get a reference to the gst_metadata
        # instance. This a cheap - UGLY - way to get the frontend without
        # changing a lot of client code. It is really ugly as we assume there
        # is only one frontend, which might not be the case in the future...
        frontend = common.application.interface_controller.frontends.values()[0]
        # Retrieve and store a reference to gst_metadata
        controllers = frontend.retrieve_controllers('/poblesec')
        try:
            self.gst_metadata = controllers[0].gst_metadata
        except AttributeError:
            msg = 'GstMetadata missing: thumbnails will not be generated.'
            self.warning(msg)
            self.gst_metadata = None

    def _updated_thumbnail(self, thumbnail, item):
        item.thumbnail_uri = unicode(thumbnail)
        return thumbnail

    def _request_thumbnail(self, path):
        if self.gst_metadata is None:
            msg = 'No gst_metadata.'
            return defer.fail(IncompleteMetadataResponse(msg))

        def got_metadata(metadata):
            return metadata['thumbnail']

        metadata = {'uri': MediaUri("file://%s" % path), 'thumbnail': None,}

        dfr = self.gst_metadata.get_metadata(metadata)
        dfr.addCallback(got_metadata)
        return dfr

class VideoViewMode(GenericListViewMode, ThumbnailMixin):
    
    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        return defer.succeed(os.path.basename(item.file_path))

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        resource = 'elisa.plugins.poblesec.file_video'
        return resource
    
    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None
        if hasattr(item, 'thumbnail_uri') and item.thumbnail_uri != None:
            return defer.succeed(item.thumbnail_uri)
        
        dfr = self._request_thumbnail(item.file_path)
        dfr.addCallback(self._updated_thumbnail, item)
        return dfr
    
    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        if hasattr(item, 'thumbnail_uri') and item.thumbnail_uri != None:
            return item.thumbnail_uri
        else:
            return None

class VideoController(HierarchyController):
    
    empty_label = _('There are no videos in this section')
    empty_icon = 'elisa.plugins.poblesec.file_video'

    def initialize(self):
        dfr = super(VideoController, self).initialize()
        self.actions.extend(self.make_actions())
        return dfr
    
    def node_clicked(self, widget, item):
        if isinstance(item, Action):
            dfr = item.run()
        elif hasattr(item, 'action') and issubclass(item.action, Action):
            action = item.action(self)
            dfr = action.run(item)
        else:
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            history = controllers[0].history
            # We are using the basename of the file_path as the name parameter this should be changed when we can get a proper name
            dfr = history.append_controller('/poblesec/database/video/track', \
                                             os.path.basename(item.file_path), \
                                             name=os.path.basename(item.file_path), \
                                             file_path=item.file_path)
         
        return dfr
    
    def make_actions(self):
        actions = []
        
        action = VideoPlayAllAction(self)
        action.title = _('Play All')
        action.subtitle = _('Play All Videos')
        actions.append(action)

        action = VideoShuffleAction(self)
        action.title = _('Shuffle & Play')
        action.subtitle = _('Shuffle & Play All Videos')
        actions.append(action)
        
        return actions

class AllMixin(object):
    def _load(self, result, filter):
        self.filter = filter

        def got_items(models):
            self.model.extend(models)
            return self

        def get_items(result_set):
            return result_set.all()
        
        def sort_by_name(result_set):
            result_set.order_by(Video.name, Video.file_path)
            return result_set

        store = common.application.store

        dfr = store.find(filter, filter.file_path == File.path)
        
        dfr.addCallback(sort_by_name)

        dfr.addCallback(get_items)
        dfr.addCallback(got_items)
        return dfr
        
class AllVideosController(AllMixin, VideoController):
    
    def initialize(self):
        dfr = super(AllVideosController, self).initialize()

        dfr.addCallback(self._load, Video)
        return dfr
        
class VideosMonthController(VideoController):

    empty_label = _('There are no videos in this section')
    empty_icon = 'elisa.plugins.poblesec.by_decade'

    def initialize(self, month=None, year=None):
        self.date = datetime.datetime(year, month, 1)
        
        dfr = super(VideosMonthController, self).initialize()

        def get_videos(result):
            store = common.application.store
            try:
                last = datetime.datetime(year, month+1, 1)
            except ValueError:
                # we reached the month 12 + 1, so we have to skip to
                # january next year
                last = datetime.datetime(year+1, 1, 1)

            dfr = store.find(Video, Video.creation_time >= self.date,
                             Video.creation_time < last,
                             Video.file_path == File.path,
                             File.hidden == False)
            return dfr

        def got_videos(videos):
            self.model.extend(videos)
            return self

        def sort_videos(result_set):
            result_set.order_by(Video.creation_time, Video.file_path)
            return result_set.all()

        dfr.addCallback(get_videos)
        dfr.addCallback(sort_videos)
        dfr.addCallback(got_videos)

        return dfr
    
    def make_actions(self):
        actions = []
        
        action = VideoPlayAllAction(self)
        action.title = _('Play All')
        subtitle = self.date.strftime(_("Videos from %B, %Y"))
        action.subtitle = subtitle
        actions.append(action)
        
        action = VideoShuffleAction(self)
        action.title = _('Shuffle and Play')
        subtitle = self.date.strftime(_("Videos from %B, %Y"))
        action.subtitle = subtitle
        actions.append(action)
        
        return actions
        
class VideoTimesViewMode(GenericListViewMode):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)

        return defer.succeed(item.strftime(_('%B, %Y')))

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        resource = 'elisa.plugins.poblesec.video_folders'
        return resource

    def get_image(self, item, theme):
        return None

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        return None
        
class VideosTimesController(HierarchyController):

    query = "SELECT DISTINCT STRFTIME('%m %Y', creation_time)" \
            " from videos, files where videos.creation_time not NULL" \
            " and files.path = videos.file_path and files.hidden = 0" \
            " order by creation_time;"

    empty_label = _('There are no videos in this section')
    empty_icon = 'elisa.plugins.poblesec.by_decade'

    def initialize(self):
        dfr = super(VideosTimesController, self).initialize()

        def get_times(result):
            store = common.application.store
            dfr = store.execute(self.query)
            return dfr

        def get_all(result):
            return result.get_all()

        def iterate(rows):
            for row in rows:
                month, year = row[0].split()
                date = datetime.datetime(int(year), int(month), 1)
                self.model.append(date)
                yield date

        def got_times(times):
            return task.coiterate(iterate(times))

        def done(res):
            return self

        dfr.addCallback(get_times)
        dfr.addCallback(get_all)
        dfr.addCallback(got_times)
        dfr.addCallback(done)
        return dfr

    def make_actions(self):
        actions = []

        action = VideoPlayAllAction(self)
        action.title = _('Play All')
        action.sub_title = _('View all videos')
        actions.append(action)

        return actions

    def node_clicked(self, widget, model):
        if isinstance(model, Action):
            model.run()
        else:
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            history = controllers[0].history
            dfr = history.append_controller('/poblesec/database/video/by_month',
                    model.strftime(_('%B, %Y')), year=model.year, \
                    month=model.month)

class VideoTrackViewMode(VideoViewMode):

    def get_sublabel(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.subtitle)

                        
class VideoTrackController(VideoController):

    def initialize(self, name=None, file_path=None):
        self.name = name
        self.file_path = file_path
        dfr = super(VideoTrackController, self).initialize()
        return dfr
    
    def is_empty(self):
        return super(VideoTrackController, self).is_empty() and self.file_path is None

    def make_actions(self):
        actions = []
        
        action = VideoPlayAllAction(self)
        action.title = _('Play Video')
        action.subtitle = _("Play '%s'") % self.name
        actions.append(action)

        action = VideoAddToFavoritesAction(self)
        action.untoggled_title = _('Add to Favorites')
        action.untoggled_subtitle = _("Add '%s' To Your Favorites") % self.name
        action.toggled_title = _('Remove From Favorites')
        action.toggled_subtitle = _("Remove '%s' From Your Favorites") % self.name
        actions.append(action)

        dfr = common.application.store.find(FavoritesItem,
                                            FavoritesItem.foreign_id == self.file_path,
                                            FavoritesItem.foreign_class == u'Video')
        dfr.addCallback(lambda rs: rs.all())
        dfr.addCallback(action.setup)
        
        return actions

# All Videos
class AllVideosVerticalWithPreview(AllVideosController, MenuItemPreviewListController):
    fastscroller = True
    view_mode = VideoViewMode
    
    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class AllVideosCoverflow(AllVideosController, ImageWithReflectionCoverflowController):
    view_mode = VideoViewMode

class AllVideosGrid(AllVideosController, GridItemGridController):
    view_mode = VideoViewMode

class AllVideosListSwitcherController(ListSwitcherController):
    modes = [AllVideosVerticalWithPreview,
             AllVideosCoverflow,
             AllVideosGrid]
    default_mode = AllVideosVerticalWithPreview
    
# Videos by month
class VideosMonthVerticalWithPreview(VideosMonthController, MenuItemPreviewListController):
    view_mode = VideoViewMode
    
    def item_to_label(self, item):
        if isinstance(item, Action):
            return '#'
        return item.name

class VideosMonthCoverflow(VideosMonthController, ImageWithReflectionCoverflowController):
    view_mode = VideoViewMode

class VideosMonthGrid(VideosMonthController, GridItemGridController):
    view_mode = VideoViewMode

class VideosMonthListSwitcherController(ListSwitcherController):
    modes = [VideosMonthVerticalWithPreview,
             VideosMonthCoverflow,
             VideosMonthGrid]
    default_mode = VideosMonthVerticalWithPreview
    
# Video Times Overview
class VideoTimesVerticalWithPreview(VideosTimesController, MenuItemPreviewListController):
    view_mode = VideoTimesViewMode

class VideoTimesCoverflow(VideosTimesController, ImageWithReflectionCoverflowController):
    view_mode = VideoTimesViewMode

class VideoTimesGrid(VideosTimesController, GridItemGridController):
    view_mode = VideoTimesViewMode

class VideoTimesListSwitcherController(ListSwitcherController):
    modes = [VideoTimesVerticalWithPreview,
             VideoTimesCoverflow,
             VideoTimesGrid]
    default_mode = VideoTimesVerticalWithPreview
    
# Videos tracks
class VideoTrackVerticalWithPreview(VideoTrackController, DoubleLineMenuItemPreviewListController):
    view_mode = VideoTrackViewMode

class VideoTrackCoverflow(VideoTrackController, ImageWithReflectionCoverflowController):
    view_mode = VideoTrackViewMode

class VideoTrackGrid(VideoTrackController, GridItemGridController):
    view_mode = VideoTrackViewMode

class VideoTrackListSwitcherController(ListSwitcherController):
    modes = [VideoTrackVerticalWithPreview,
             VideoTrackCoverflow,
             VideoTrackGrid]
    default_mode = VideoTrackVerticalWithPreview

