/*
 * Copyright (C) 2007 Intel
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <njp@o-hand.com>
 *             Neil Jagdish Patel <neil.patel@canonical.com>
 *               - Updated to use default GNOME gconf keys
 *               - Add support for blurring the background (threaded)
 *               - Other cusomisations to fit with the launcher
 */

#include <glib.h>
#include <gio/gio.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include <gtk/gtk.h>

#include "cairo-utils.h"
#include "launcher-background.h"
#include "launcher-behave.h"
#include "launcher-config.h"
#include "launcher-defines.h"
#include "launcher-util.h"
#include "launcher-wm.h"

G_DEFINE_TYPE (LauncherBackground, launcher_background, CLUTTER_TYPE_GROUP)

#define LAUNCHER_BACKGROUND_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
        LAUNCHER_TYPE_BACKGROUND, LauncherBackgroundPrivate))

/* Gconf keys */
#define BG_PATH    "/desktop/gnome/background"
#define BG_FILE    BG_PATH "/picture_filename" /* string */
#define BG_OPTION  BG_PATH "/picture_options" 
                   /* none|wallpaper|centred|scaled|stretched|zoom */

#define BG_DEFAULT  PKGDATADIR "/default.svg"

struct _LauncherBackgroundPrivate
{
  ClutterActor *texture;
  ClutterActor *blurred;
  ClutterActor *fade;
  
  gchar *filename;
  gchar *option;
  gchar *old_hash;

  ClutterTimeline *timeline;
  ClutterEffectTemplate *temp;
  ClutterAlpha *alpha;
  ClutterBehaviour *behave;

  GThread *thread;
  guint timeout;

  gboolean fade_out;

  gint x; gint y;
};

typedef struct 
{
  LauncherBackground *bg;
  
  GdkPixbuf *pixbuf;
  gchar     *filename;
  gchar     *option;

} BgClosure;

/* 
 * Just load the file normally,if its larger than the stage, clip it */
static void
load_normal (const gchar *filename, ClutterActor *texture)
{
  GdkPixbuf *pixbuf = NULL;
  GError *error = NULL;
  gint width, height;
  gint subw = 0, subh = 0; 

  pixbuf = gdk_pixbuf_new_from_file (filename, &error);

  if (error)
  {
    g_warning ("%s", error->message);
    g_error_free (error);
    return;
  }

  width = gdk_pixbuf_get_width (pixbuf);
  height = gdk_pixbuf_get_height (pixbuf);

  if (width > CLUTTER_STAGE_WIDTH ())
    subw = CLUTTER_STAGE_WIDTH ();
  else
    subw = width;

  if (height > CLUTTER_STAGE_HEIGHT ())
    subh = CLUTTER_STAGE_HEIGHT ();
  else
    subh = height;

  if (subw && subh)
  {
    GdkPixbuf *temp = pixbuf;

    pixbuf = gdk_pixbuf_new_subpixbuf (temp, 0, 0, subw, subh);
    g_object_unref (temp);
  }

  clutter_texture_set_pixbuf (CLUTTER_TEXTURE (texture), pixbuf, NULL);
  clutter_actor_set_size (texture, 
                          CLUTTER_STAGE_WIDTH (),
                          CLUTTER_STAGE_HEIGHT ());
  clutter_actor_set_position (texture, 0, 0);

  g_object_unref (pixbuf);
}

/*
 * Simply return a stretched version of the image
 */
static void
load_stretched (const gchar *filename, ClutterActor *texture)
{
  GdkPixbuf *pixbuf = NULL;
  GError *error = NULL;

  pixbuf = gdk_pixbuf_new_from_file_at_scale (filename,
                                              CLUTTER_STAGE_WIDTH (),
                                              CLUTTER_STAGE_HEIGHT (),
                                              FALSE,
                                              &error);
  if (error)
  {
    g_warning ("%s", error->message);
    g_error_free (error);
  }

  clutter_texture_set_pixbuf (CLUTTER_TEXTURE (texture), pixbuf, NULL);
  clutter_actor_set_size (texture, 
                          CLUTTER_STAGE_WIDTH (),
                          CLUTTER_STAGE_HEIGHT ());
  clutter_actor_set_position (texture, 0, 0);

  g_object_unref (pixbuf);
}

/*
 * Center the image on the stage
 */
static void
load_centred (const gchar *filename, ClutterActor *texture)
{
  GdkPixbuf *pixbuf = NULL;
  GError *error = NULL;
  gint w, h, x, y;

  pixbuf = gdk_pixbuf_new_from_file (filename, &error);

  if (error)
  {
    g_warning ("%s", error->message);
    g_error_free (error);
    return;
  }

  clutter_texture_set_pixbuf (CLUTTER_TEXTURE (texture), pixbuf, NULL);
  
  w = gdk_pixbuf_get_width (pixbuf);
  h = gdk_pixbuf_get_height (pixbuf);
  x = (CLUTTER_STAGE_WIDTH ()/2) - (w/2);
  y = (CLUTTER_STAGE_HEIGHT ()/2) - (h/2);

  clutter_actor_set_size (texture, w, h);
  clutter_actor_set_position (texture, x, y);  

  g_object_unref (pixbuf);
}

/*
 * Load the image scaled with the correct aspect ratio 
 */
static void
load_scaled (const gchar *filename, ClutterActor *texture)
{
  GdkPixbuf *pixbuf = NULL;
  GError *error = NULL;
  gint w, h, x, y;
  
  pixbuf = gdk_pixbuf_new_from_file_at_scale (filename,
                                              CLUTTER_STAGE_WIDTH (),
                                              CLUTTER_STAGE_HEIGHT (),
                                              TRUE,
                                              &error);
  if (error)
  {
    g_warning ("%s", error->message);
    g_error_free (error);
  }

  if (gdk_pixbuf_get_width (pixbuf) < CLUTTER_STAGE_WIDTH ())
  {
    GdkPixbuf *temp = gdk_pixbuf_new_from_file_at_scale (filename,
                                                         CLUTTER_STAGE_WIDTH(),
                                                         -1,
                                                         TRUE,
                                                         NULL);
    g_object_unref (pixbuf);
    pixbuf = temp;
  }
  else if (gdk_pixbuf_get_height (pixbuf) < CLUTTER_STAGE_HEIGHT ())
  {
    GdkPixbuf *temp = gdk_pixbuf_new_from_file_at_scale (filename,
                                                         -1,
                                                         CLUTTER_STAGE_HEIGHT(),
                                                         TRUE,
                                                         NULL);
    g_object_unref (pixbuf);
    pixbuf = temp;  
  }
  
  clutter_texture_set_pixbuf (CLUTTER_TEXTURE (texture), pixbuf, NULL);
  
  w = gdk_pixbuf_get_width (pixbuf);
  h = gdk_pixbuf_get_height (pixbuf);
  x = (CLUTTER_STAGE_WIDTH ()/2) - (w/2);
  y = (CLUTTER_STAGE_HEIGHT ()/2) - (h/2);

  clutter_actor_set_size (texture, w, h);
  clutter_actor_set_position (texture, x, y);  

  g_object_unref (pixbuf);
}

/* 
 * Load the image, and then tile it until it covers the entire stage 
 */
static void
load_wallpaper (const gchar *filename, ClutterActor *texture)
{
  GdkPixbuf *pixbuf = NULL;
  GdkPixbuf *tiled = NULL;
  GError *error = NULL;
  gint w, h, x, y;
  gint rows, cols, r, c;

  pixbuf = gdk_pixbuf_new_from_file (filename, &error);

  if (error)
  {
    g_warning ("%s", error->message);
    g_error_free (error);
    return;
  }

  x = y = 0;
  rows = cols = 1;
  w = gdk_pixbuf_get_width (pixbuf);
  h = gdk_pixbuf_get_height (pixbuf);

  /* Find the number of rows and columns */
  if (w < CLUTTER_STAGE_WIDTH ())
    cols = (gint)ceil (CLUTTER_STAGE_WIDTH () / (gdouble)w);
  
  if (h < CLUTTER_STAGE_HEIGHT ())
    rows = (gint)ceil (CLUTTER_STAGE_HEIGHT () / (gdouble)h);

  /* Create the new pixbuf */
  tiled = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, w*cols, h*rows);

  /* For the number of rows, tile the cols */
  for (r = 0; r < rows; r++)
  {
    for (c = 0; c < cols; c++)
    {
      gdk_pixbuf_composite (pixbuf, tiled, x, y, w, h, x, y, 1, 1,
                            GDK_INTERP_BILINEAR, 255);
      x += w;
    }
    y += h;
    x = 0;
  }

  g_object_unref (pixbuf);
  
  clutter_texture_set_pixbuf (CLUTTER_TEXTURE (texture), tiled, NULL);
  clutter_actor_set_position (texture, 0, 0);
}

static gchar *
hash_blur (const gchar *path, const gchar *option)
{
  gchar *leaf;
  gchar *checksum;
  gchar *filename;

  leaf = g_strdup_printf ("%s-%s", path, option);
  checksum = g_compute_checksum_for_string (G_CHECKSUM_MD5, 
                                            leaf, strlen (leaf));

  filename = g_build_filename (g_get_home_dir (), CONFIG_DIR, 
                               checksum, NULL);

  g_free (leaf);
  g_free (checksum);
  
  return filename;
}


static void
save_blur (LauncherBackground *bg)
{
  LauncherBackgroundPrivate *priv = bg->priv;
  GdkPixbuf *pixbuf;
  gchar *filename;
  GError *error = NULL;

  if (priv->old_hash)
  {
    GFile *file = g_file_new_for_path (priv->old_hash);
    g_file_delete (file, NULL, NULL);
    g_object_unref (file);
    g_free (priv->old_hash);
  }

  g_object_get (priv->blurred, "pixbuf", &pixbuf, NULL);

  filename = hash_blur (priv->filename, priv->option);
  
  gdk_pixbuf_save (pixbuf, filename, "png", &error, NULL);
  if (error)
  {
    g_warning ("Unable to cache blurred image: %s\n", error->message);
    g_error_free (error);
  }
  priv->old_hash = filename;
  g_object_unref (pixbuf);
}


static gboolean
create_blur (LauncherBackground *bg)
{
#define BLUR_RADIUS 55
#define P_WIDTH 320
#define P_HEIGHT 240
  LauncherBackgroundPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  GdkPixbuf *pixbuf, *blurred;
  cairo_surface_t *surface;
  cairo_t *cr;

  g_return_val_if_fail (LAUNCHER_IS_BACKGROUND (bg), FALSE);
  priv = bg->priv;

  pixbuf = clutter_texture_get_pixbuf (CLUTTER_TEXTURE (priv->texture));

  if (!GDK_IS_PIXBUF (pixbuf))
    return FALSE;

  surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
                                   gdk_pixbuf_get_width (pixbuf),
                                   gdk_pixbuf_get_height (pixbuf));

  cr = cairo_create (surface);
  cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
  cairo_fill (cr);
  cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
  gdk_cairo_set_source_pixbuf (cr, pixbuf, 0, 0);
  cairo_paint (cr);
  
  cairo_image_surface_blur (cairo_get_target (cr), BLUR_RADIUS, BLUR_RADIUS);
  blurred = launcher_util_get_pixbuf_from_surface (cairo_get_target (cr));
  
  cairo_destroy (cr);

  /* If we're on Poulsbo, try and reduce the texture size */
  if (cfg->is_poulsbo)
  {
    GdkPixbuf *temp = blurred;
    blurred = gdk_pixbuf_scale_simple (temp, P_WIDTH, P_HEIGHT, 
                                       GDK_INTERP_BILINEAR);
    clutter_texture_set_pixbuf (CLUTTER_TEXTURE (priv->blurred), blurred, NULL);
    clutter_actor_set_scale (priv->blurred, CSW()/P_WIDTH,  CSH()/P_HEIGHT);
    clutter_actor_set_anchor_point_from_gravity (priv->blurred,
                                                 CLUTTER_GRAVITY_CENTER);
    clutter_actor_set_position (priv->blurred, CSW()/2, CSH()/2);
    
    g_object_unref (temp);
  }
  else
  {
    clutter_texture_set_pixbuf (CLUTTER_TEXTURE (priv->blurred), blurred, NULL);
  }

  save_blur (bg);

  g_object_unref (blurred);
  g_object_unref (pixbuf);
  cairo_surface_destroy (surface);

  return FALSE;
}

static gboolean 
load_blur (LauncherBackground *bg)
{
  LauncherBackgroundPrivate *priv = bg->priv;
  GdkPixbuf *pixbuf;
  gchar *filename;
  
  filename = hash_blur (priv->filename, priv->option);

  pixbuf = gdk_pixbuf_new_from_file (filename, NULL);

  if (GDK_IS_PIXBUF (pixbuf))
  {
    clutter_texture_set_pixbuf (CLUTTER_TEXTURE (priv->blurred), pixbuf, NULL);
    priv->old_hash = filename;
    g_object_unref (pixbuf);
    return TRUE;
  }

  g_free (filename);
  return FALSE;
}

static gboolean
on_blur_made (BgClosure *data)
{
  LauncherBackgroundPrivate *priv = data->bg->priv;

  g_debug ("Blurring thread finished");

  if (data->filename && data->option && 
      strcmp (data->filename,priv->filename) == 0  &&
      strcmp (data->option, priv->option) == 0)
  {
    gdouble scale = 0;
    
    clutter_texture_set_pixbuf (CLUTTER_TEXTURE (priv->blurred), 
                                data->pixbuf, NULL);
    clutter_actor_get_scale (CLUTTER_ACTOR (data->bg), &scale, &scale);
    
    if (scale > 1.1)
    {
      clutter_actor_set_opacity (priv->blurred, 255);
    }
    
    save_blur (data->bg);

    priv->timeout = 0;
    priv->thread = NULL;  
  }

  g_object_unref (data->pixbuf);
  g_free (data->filename);
  g_free (data->option);
  g_free (data);

  return FALSE;
}

static gpointer
threaded_blur (BgClosure *data)
{
#define BLUR_RADIUS 55
#define P_WIDTH 320
#define P_HEIGHT 240
  GdkPixbuf *pixbuf, *blurred;
  cairo_surface_t *surface;
  cairo_t *cr;

  g_debug ("Blurring thread started");

  pixbuf = data->pixbuf;

  if (!GDK_IS_PIXBUF (pixbuf))
    return FALSE;

  surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
                                   gdk_pixbuf_get_width (pixbuf),
                                   gdk_pixbuf_get_height (pixbuf));

  cr = cairo_create (surface);
  cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
  cairo_fill (cr);
  cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
  gdk_cairo_set_source_pixbuf (cr, pixbuf, 0, 0);
  cairo_paint (cr);
  
  cairo_image_surface_blur (cairo_get_target (cr), BLUR_RADIUS, BLUR_RADIUS);
  blurred = launcher_util_get_pixbuf_from_surface (cairo_get_target (cr));
  
  cairo_destroy (cr);

  g_object_unref (data->pixbuf);
  data->pixbuf = blurred;

  clutter_threads_add_idle ((GSourceFunc)on_blur_made, data);

  cairo_surface_destroy (surface);

  return NULL;
}

static gboolean
make_blur_timout (LauncherBackground *bg)
{
  LauncherBackgroundPrivate *priv = bg->priv;
  BgClosure *data = g_new0 (BgClosure, 1);

  /* It doesn't matter is there is already a thread running as when it finshes
   * it will realise that the picture or/and options have changed and therefore
   * discard it's changes.
   *
   * This is all because we can't cancel a thread
   */

  data->bg = bg;
  data->filename = g_strdup (priv->filename);
  data->option = g_strdup (priv->option);
  data->pixbuf = clutter_texture_get_pixbuf (CLUTTER_TEXTURE (priv->texture));

  priv->thread = g_thread_create ((GThreadFunc)threaded_blur, data, TRUE, NULL);

  return FALSE;
}

/* 
 * Use the filename and option values to create a background pixbuf, and set 
 * the internal tetxures pixbuf.
 * We try and get the smallest possible pixbuf to make sure we don't abuse
 * texture memory.
 */
static void
ensure_layout (LauncherBackground *bg)
{
  LauncherBackgroundPrivate *priv;
      
  g_return_if_fail (LAUNCHER_IS_BACKGROUND (bg));
  priv = bg->priv;

  if (priv->filename == NULL 
        || strcmp (priv->filename, "default") == 0
        || !g_file_test (priv->filename, G_FILE_TEST_EXISTS))
    priv->filename = g_strdup (BG_DEFAULT);

  if (priv->option == NULL)
    priv->option = g_strdup ("stretched");

  if (priv->option == NULL || strcmp (priv->option, "none") == 0)
  {
    load_normal (priv->filename, priv->texture);
  }
  else if (strcmp (priv->option, "wallpaper") == 0)
  {
    load_wallpaper (priv->filename, priv->texture);
  }
  else if (strcmp (priv->option, "centred") == 0)
  {
    load_centred (priv->filename, priv->texture);
  }
  else if (strcmp (priv->option, "scaled") == 0)
  {
    load_scaled (priv->filename, priv->texture);
  }
  else if (strcmp (priv->option, "zoom") == 0)
  {
    load_scaled (priv->filename, priv->texture);
  }
  else /* stretched */
  {
    load_stretched (priv->filename, priv->texture);
  }
  
  if (!load_blur (bg))
  {
    clutter_actor_set_opacity (priv->blurred, 0);
    clutter_texture_set_pixbuf (CLUTTER_TEXTURE (priv->blurred), NULL, NULL);
    if (priv->timeout)
      g_source_remove (priv->timeout);
    
    priv->timeout = g_timeout_add (1000, (GSourceFunc)make_blur_timout, bg); 
  }
  clutter_actor_queue_redraw (CLUTTER_ACTOR (bg));
}

/*
 * The animation function. At the moment, we just to a simple fade in/fade out
 */
static void
alpha_func (ClutterBehaviour   *behave,
            guint32             alpha_value,
            LauncherBackground *bg)
{
  LauncherBackgroundPrivate *priv;
  gfloat factor;

  g_return_if_fail (LAUNCHER_IS_BACKGROUND (bg));
  priv = bg->priv;

  factor = (gfloat)alpha_value / CLUTTER_ALPHA_MAX_ALPHA;
  
  if (priv->fade_out)
    clutter_actor_set_opacity (CLUTTER_ACTOR (bg), 255 - (255*factor));
  else
    clutter_actor_set_opacity (CLUTTER_ACTOR (bg), 255 * factor);

  clutter_actor_queue_redraw (CLUTTER_ACTOR (bg));
}

static void
on_timeline_completed (ClutterTimeline *timeline, LauncherBackground *bg)
{
  LauncherBackgroundPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_BACKGROUND (bg));
  priv = bg->priv;

  if (priv->fade_out)
  {
    ensure_layout (bg);
    priv->fade_out = FALSE;
    clutter_timeline_start (priv->timeline);
  }
  else
  {
    priv->fade_out = TRUE;
  }
}

/* Gconf callbacks */
static void
on_bg_filename_changed (GConfClient        *client,
                        guint               cid,
                        GConfEntry         *entry,
                        LauncherBackground *bg)
{
  LauncherBackgroundPrivate *priv;
  GConfValue *value = NULL;

  g_return_if_fail (LAUNCHER_IS_BACKGROUND (bg));
  priv = bg->priv;

  value = gconf_entry_get_value (entry);

  if (priv->filename 
      && strcmp (priv->filename, gconf_value_get_string (value)) == 0)
    return;


  if (priv->filename)
    g_free (priv->filename);

  priv->filename = g_strdup (gconf_value_get_string (value));

  clutter_timeline_start (priv->timeline);
}

static void
on_bg_option_changed (GConfClient        *client,
                      guint               cid,
                      GConfEntry         *entry,
                      LauncherBackground *bg)
{
  LauncherBackgroundPrivate *priv;
  GConfValue *value = NULL;

  g_return_if_fail (LAUNCHER_IS_BACKGROUND (bg));
  priv = bg->priv;

  value = gconf_entry_get_value (entry);

  if (priv->option && strcmp (priv->option, gconf_value_get_string (value))==0)
    return;

  if (priv->option)
    g_free (priv->option);

  priv->option = g_strdup (gconf_value_get_string (value));

  clutter_timeline_start (priv->timeline);
}

static void
change_background (GtkMenuItem *item, gpointer data)
{
#define BG_EXEC "gnome-appearance-properties --show-page=background"

  gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                   BG_EXEC, NULL);
}

static void
position_func (GtkMenu  *menu, 
               gint     *x,
               gint     *y, 
               gboolean *push_in,
               gpointer  data)
{
  LauncherBackgroundPrivate *priv;

  priv = LAUNCHER_BACKGROUND_GET_PRIVATE (data);

  *x = priv->x;
  *y = priv->y;
  *push_in = TRUE;
}

static gboolean
on_click (ClutterActor *texture, 
          ClutterButtonEvent *event, 
          LauncherBackground *bg)
{
  GtkWidget *menu;
  GtkWidget *item;

  if (event->button != 3)
    return FALSE;

  bg->priv->x = event->x;
  bg->priv->y = event->y;

  menu = gtk_menu_new ();

  item = gtk_menu_item_new_with_label ("Change Desktop Background");
  gtk_widget_show (item);
  g_signal_connect (item, "activate",
                    G_CALLBACK (change_background), bg);

  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  gtk_menu_popup (GTK_MENU (menu),
                  NULL, NULL, 
                  position_func, bg,
                  3, event->time);

  return TRUE;
}
void      
launcher_background_blur (LauncherBackground *bg)
{
  LauncherBackgroundPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_BACKGROUND (bg));
  priv = bg->priv;

  clutter_effect_fade (priv->temp, priv->blurred, 255, NULL, NULL);
  clutter_effect_scale (priv->temp, CLUTTER_ACTOR (bg), 1.2, 1.2, NULL, NULL);
}

void      
launcher_background_unblur (LauncherBackground *bg)
{
  LauncherBackgroundPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_BACKGROUND (bg));
  priv = bg->priv;

  clutter_effect_fade (priv->temp, priv->blurred, 0, NULL, NULL);
  clutter_effect_scale (priv->temp, CLUTTER_ACTOR (bg), 
                        1.01, 1.01, NULL, NULL);
}



/* GObject functions */
static void
launcher_background_dispose (GObject *object)
{
  G_OBJECT_CLASS (launcher_background_parent_class)->dispose (object);
}

static void
launcher_background_finalize (GObject *background)
{
  LauncherBackgroundPrivate *priv;
  
  g_return_if_fail (LAUNCHER_IS_BACKGROUND (background));
  priv = LAUNCHER_BACKGROUND (background)->priv;

  g_free (priv->filename);
  g_free (priv->option);
  g_free (priv->old_hash);

  G_OBJECT_CLASS (launcher_background_parent_class)->finalize (background);
}


static void
launcher_background_class_init (LauncherBackgroundClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = launcher_background_finalize;
  obj_class->dispose = launcher_background_dispose;

  g_type_class_add_private (obj_class, sizeof (LauncherBackgroundPrivate)); 
}

static void
launcher_background_init (LauncherBackground *background)
{
#define LAUNCHER_LOWBG "LAUNCHER_LOWBG"
  LauncherBackgroundPrivate *priv;
  GConfClient *client = gconf_client_get_default ();

  priv = background->priv = LAUNCHER_BACKGROUND_GET_PRIVATE (background);
  priv->fade_out = TRUE;
  priv->filename = priv->option = priv->old_hash = NULL;

  priv->texture = g_object_new (CLUTTER_TYPE_TEXTURE, 
                                "tiled", TRUE, NULL);
  clutter_container_add_actor (CLUTTER_CONTAINER (background), priv->texture);
  clutter_actor_set_size (priv->texture, 
                          CLUTTER_STAGE_WIDTH (),
                          CLUTTER_STAGE_HEIGHT ());
  clutter_actor_set_position (priv->texture, 0, 0);
  clutter_actor_show (priv->texture);

  clutter_actor_set_reactive (priv->texture, TRUE);
  g_signal_connect (priv->texture, "button-release-event", 
                    G_CALLBACK (on_click), background);

  priv->blurred = g_object_new (CLUTTER_TYPE_TEXTURE, 
                                "tiled", TRUE, NULL);  clutter_container_add_actor (CLUTTER_CONTAINER (background), priv->blurred);
  clutter_actor_set_size (priv->blurred, 
                          CLUTTER_STAGE_WIDTH (),
                          CLUTTER_STAGE_HEIGHT ());
  clutter_actor_set_position (priv->blurred, 0, 0);
  clutter_actor_set_opacity (priv->blurred, 0);
  clutter_actor_show (priv->blurred);

  clutter_actor_set_reactive (priv->blurred, TRUE);
  g_signal_connect (priv->blurred, "button-release-event", 
                    G_CALLBACK (on_click), background);

  if (g_getenv (LAUNCHER_LOWBG))
  {
    g_object_set (priv->texture, "filter-quality", 0, NULL);
    g_object_set (priv->blurred, "filter-quality", 0, NULL);
  }

  gconf_client_add_dir (client, BG_PATH, GCONF_CLIENT_PRELOAD_NONE, NULL);

  priv->filename = g_strdup (gconf_client_get_string (client, BG_FILE, NULL));
  gconf_client_notify_add (client, BG_FILE,
                           (GConfClientNotifyFunc)on_bg_filename_changed,
                           background, NULL, NULL);
  
  priv->option = g_strdup (gconf_client_get_string (client, BG_OPTION, NULL));
  gconf_client_notify_add (client, BG_OPTION,
                           (GConfClientNotifyFunc)on_bg_option_changed,
                           background, NULL, NULL);
  

  priv->timeline = launcher_util_timeline_new_fast ();
  priv->temp = clutter_effect_template_new (priv->timeline, 
                                            clutter_sine_inc_func);
  priv->alpha = clutter_alpha_new_full (priv->timeline,
                                        clutter_sine_inc_func,
                                        NULL, NULL);
  priv->behave = launcher_behave_new (priv->alpha, 
                                      (LauncherBehaveAlphaFunc)alpha_func,
                                      (gpointer)background);

  g_signal_connect (priv->timeline, "completed",
                    G_CALLBACK (on_timeline_completed), (gpointer)background);

  ensure_layout (background);
  if (!load_blur (background))
    create_blur (background);

  g_object_unref (client);
}

ClutterActor *
launcher_background_new (void)
{
  ClutterActor *background;

  background = g_object_new (LAUNCHER_TYPE_BACKGROUND,
                       NULL);
  return background;
}
